/// Helper class for country code validation and country information
class CountryHelper {
  /// Valid ISO alpha-2 country codes
  static const Set<String> _validCountryCodes = {
    'ad', 'ae', 'af', 'ag', 'ai', 'al', 'am', 'ao', 'aq', 'ar', 'as', 'at',
    'au', 'aw', 'ax', 'az', 'ba', 'bb', 'bd', 'be', 'bf', 'bg', 'bh', 'bi',
    'bj', 'bl', 'bm', 'bn', 'bo', 'bq', 'br', 'bs', 'bt', 'bv', 'bw', 'by',
    'bz', 'ca', 'cc', 'cd', 'cf', 'cg', 'ch', 'ci', 'ck', 'cl', 'cm', 'cn',
    'co', 'cr', 'cu', 'cv', 'cw', 'cx', 'cy', 'cz', 'de', 'dj', 'dk', 'dm',
    'do', 'dz', 'ec', 'ee', 'eg', 'eh', 'er', 'es', 'et', 'fi', 'fj', 'fk',
    'fm', 'fo', 'fr', 'ga', 'gb', 'gd', 'ge', 'gf', 'gg', 'gh', 'gi', 'gl',
    'gm', 'gn', 'gp', 'gq', 'gr', 'gs', 'gt', 'gu', 'gw', 'gy', 'hk', 'hm',
    'hn', 'hr', 'ht', 'hu', 'id', 'ie', 'il', 'im', 'in', 'io', 'iq', 'ir',
    'is', 'it', 'je', 'jm', 'jo', 'jp', 'ke', 'kg', 'kh', 'ki', 'km', 'kn',
    'kp', 'kr', 'kw', 'ky', 'kz', 'la', 'lb', 'lc', 'li', 'lk', 'lr', 'ls',
    'lt', 'lu', 'lv', 'ly', 'ma', 'mc', 'md', 'me', 'mf', 'mg', 'mh', 'mk',
    'ml', 'mm', 'mn', 'mo', 'mp', 'mq', 'mr', 'ms', 'mt', 'mu', 'mv', 'mw',
    'mx', 'my', 'mz', 'na', 'nc', 'ne', 'nf', 'ng', 'ni', 'nl', 'no', 'np',
    'nr', 'nu', 'nz', 'om', 'pa', 'pe', 'pf', 'pg', 'ph', 'pk', 'pl', 'pm',
    'pn', 'pr', 'ps', 'pt', 'pw', 'py', 'qa', 're', 'ro', 'rs', 'ru', 'rw',
    'sa', 'sb', 'sc', 'sd', 'se', 'sg', 'sh', 'si', 'sj', 'sk', 'sl', 'sm',
    'sn', 'so', 'sr', 'ss', 'st', 'sv', 'sx', 'sy', 'sz', 'tc', 'td', 'tf',
    'tg', 'th', 'tj', 'tk', 'tl', 'tm', 'tn', 'to', 'tr', 'tt', 'tv', 'tw',
    'tz', 'ua', 'ug', 'um', 'us', 'uy', 'uz', 'va', 'vc', 've', 'vg', 'vi',
    'vn', 'vu', 'wf', 'ws', 'ye', 'yt', 'za', 'zm', 'zw'
  };

  /// Country names mapping
  static const Map<String, String> _countryNames = {
    'ad': 'Andorra', 'ae': 'United Arab Emirates', 'af': 'Afghanistan',
    'ag': 'Antigua and Barbuda', 'ai': 'Anguilla', 'al': 'Albania',
    'am': 'Armenia', 'ao': 'Angola', 'aq': 'Antarctica', 'ar': 'Argentina',
    'as': 'American Samoa', 'at': 'Austria', 'au': 'Australia', 'aw': 'Aruba',
    'ax': 'Åland Islands', 'az': 'Azerbaijan', 'ba': 'Bosnia and Herzegovina',
    'bb': 'Barbados', 'bd': 'Bangladesh', 'be': 'Belgium', 'bf': 'Burkina Faso',
    'bg': 'Bulgaria', 'bh': 'Bahrain', 'bi': 'Burundi', 'bj': 'Benin',
    'bl': 'Saint Barthélemy', 'bm': 'Bermuda', 'bn': 'Brunei', 'bo': 'Bolivia',
    'bq': 'Caribbean Netherlands', 'br': 'Brazil', 'bs': 'Bahamas',
    'bt': 'Bhutan', 'bv': 'Bouvet Island', 'bw': 'Botswana', 'by': 'Belarus',
    'bz': 'Belize', 'ca': 'Canada', 'cc': 'Cocos Islands', 'cd': 'Congo',
    'cf': 'Central African Republic', 'cg': 'Congo', 'ch': 'Switzerland',
    'ci': 'Côte d\'Ivoire', 'ck': 'Cook Islands', 'cl': 'Chile', 'cm': 'Cameroon',
    'cn': 'China', 'co': 'Colombia', 'cr': 'Costa Rica', 'cu': 'Cuba',
    'cv': 'Cape Verde', 'cw': 'Curaçao', 'cx': 'Christmas Island',
    'cy': 'Cyprus', 'cz': 'Czech Republic', 'de': 'Germany', 'dj': 'Djibouti',
    'dk': 'Denmark', 'dm': 'Dominica', 'do': 'Dominican Republic',
    'dz': 'Algeria', 'ec': 'Ecuador', 'ee': 'Estonia', 'eg': 'Egypt',
    'eh': 'Western Sahara', 'er': 'Eritrea', 'es': 'Spain', 'et': 'Ethiopia',
    'fi': 'Finland', 'fj': 'Fiji', 'fk': 'Falkland Islands', 'fm': 'Micronesia',
    'fo': 'Faroe Islands', 'fr': 'France', 'ga': 'Gabon', 'gb': 'United Kingdom',
    'gd': 'Grenada', 'ge': 'Georgia', 'gf': 'French Guiana', 'gg': 'Guernsey',
    'gh': 'Ghana', 'gi': 'Gibraltar', 'gl': 'Greenland', 'gm': 'Gambia',
    'gn': 'Guinea', 'gp': 'Guadeloupe', 'gq': 'Equatorial Guinea',
    'gr': 'Greece', 'gs': 'South Georgia', 'gt': 'Guatemala', 'gu': 'Guam',
    'gw': 'Guinea-Bissau', 'gy': 'Guyana', 'hk': 'Hong Kong', 'hm': 'Heard Island',
    'hn': 'Honduras', 'hr': 'Croatia', 'ht': 'Haiti', 'hu': 'Hungary',
    'id': 'Indonesia', 'ie': 'Ireland', 'il': 'Israel', 'im': 'Isle of Man',
    'in': 'India', 'io': 'British Indian Ocean Territory', 'iq': 'Iraq',
    'ir': 'Iran', 'is': 'Iceland', 'it': 'Italy', 'je': 'Jersey',
    'jm': 'Jamaica', 'jo': 'Jordan', 'jp': 'Japan', 'ke': 'Kenya',
    'kg': 'Kyrgyzstan', 'kh': 'Cambodia', 'ki': 'Kiribati', 'km': 'Comoros',
    'kn': 'Saint Kitts and Nevis', 'kp': 'North Korea', 'kr': 'South Korea',
    'kw': 'Kuwait', 'ky': 'Cayman Islands', 'kz': 'Kazakhstan', 'la': 'Laos',
    'lb': 'Lebanon', 'lc': 'Saint Lucia', 'li': 'Liechtenstein', 'lk': 'Sri Lanka',
    'lr': 'Liberia', 'ls': 'Lesotho', 'lt': 'Lithuania', 'lu': 'Luxembourg',
    'lv': 'Latvia', 'ly': 'Libya', 'ma': 'Morocco', 'mc': 'Monaco',
    'md': 'Moldova', 'me': 'Montenegro', 'mf': 'Saint Martin', 'mg': 'Madagascar',
    'mh': 'Marshall Islands', 'mk': 'North Macedonia', 'ml': 'Mali',
    'mm': 'Myanmar', 'mn': 'Mongolia', 'mo': 'Macao', 'mp': 'Northern Mariana Islands',
    'mq': 'Martinique', 'mr': 'Mauritania', 'ms': 'Montserrat', 'mt': 'Malta',
    'mu': 'Mauritius', 'mv': 'Maldives', 'mw': 'Malawi', 'mx': 'Mexico',
    'my': 'Malaysia', 'mz': 'Mozambique', 'na': 'Namibia', 'nc': 'New Caledonia',
    'ne': 'Niger', 'nf': 'Norfolk Island', 'ng': 'Nigeria', 'ni': 'Nicaragua',
    'nl': 'Netherlands', 'no': 'Norway', 'np': 'Nepal', 'nr': 'Nauru',
    'nu': 'Niue', 'nz': 'New Zealand', 'om': 'Oman', 'pa': 'Panama',
    'pe': 'Peru', 'pf': 'French Polynesia', 'pg': 'Papua New Guinea',
    'ph': 'Philippines', 'pk': 'Pakistan', 'pl': 'Poland', 'pm': 'Saint Pierre and Miquelon',
    'pn': 'Pitcairn Islands', 'pr': 'Puerto Rico', 'ps': 'Palestine',
    'pt': 'Portugal', 'pw': 'Palau', 'py': 'Paraguay', 'qa': 'Qatar',
    're': 'Réunion', 'ro': 'Romania', 'rs': 'Serbia', 'ru': 'Russia',
    'rw': 'Rwanda', 'sa': 'Saudi Arabia', 'sb': 'Solomon Islands',
    'sc': 'Seychelles', 'sd': 'Sudan', 'se': 'Sweden', 'sg': 'Singapore',
    'sh': 'Saint Helena', 'si': 'Slovenia', 'sj': 'Svalbard and Jan Mayen',
    'sk': 'Slovakia', 'sl': 'Sierra Leone', 'sm': 'San Marino',
    'sn': 'Senegal', 'so': 'Somalia', 'sr': 'Suriname', 'ss': 'South Sudan',
    'st': 'São Tomé and Príncipe', 'sv': 'El Salvador', 'sx': 'Sint Maarten',
    'sy': 'Syria', 'sz': 'Eswatini', 'tc': 'Turks and Caicos Islands',
    'td': 'Chad', 'tf': 'French Southern Territories', 'tg': 'Togo',
    'th': 'Thailand', 'tj': 'Tajikistan', 'tk': 'Tokelau', 'tl': 'Timor-Leste',
    'tm': 'Turkmenistan', 'tn': 'Tunisia', 'to': 'Tonga', 'tr': 'Turkey',
    'tt': 'Trinidad and Tobago', 'tv': 'Tuvalu', 'tw': 'Taiwan',
    'tz': 'Tanzania', 'ua': 'Ukraine', 'ug': 'Uganda', 'um': 'United States Minor Outlying Islands',
    'us': 'United States', 'uy': 'Uruguay', 'uz': 'Uzbekistan', 'va': 'Vatican City',
    'vc': 'Saint Vincent and the Grenadines', 've': 'Venezuela', 'vg': 'British Virgin Islands',
    'vi': 'U.S. Virgin Islands', 'vn': 'Vietnam', 'vu': 'Vanuatu',
    'wf': 'Wallis and Futuna', 'ws': 'Samoa', 'ye': 'Yemen', 'yt': 'Mayotte',
    'za': 'South Africa', 'zm': 'Zambia', 'zw': 'Zimbabwe'
  };

  /// Available flag sizes
  static const List<String> _flagSizes = ['w20', 'w40', 'w80', 'w160', 'w320', 'w640'];

  /// Validates if a country code is valid
  static bool isValidCountryCode(String countryCode) {
    return _validCountryCodes.contains(countryCode.toLowerCase());
  }

  /// Normalizes country code to lowercase
  static String normalizeCountryCode(String countryCode) {
    return countryCode.toLowerCase();
  }

  /// Gets country name from country code
  static String? getCountryName(String countryCode) {
    return _countryNames[normalizeCountryCode(countryCode)];
  }

  /// Gets all valid country codes
  static Set<String> get allCountryCodes => _validCountryCodes;

  /// Gets all country names
  static Map<String, String> get allCountryNames => _countryNames;

  /// Gets available flag sizes
  static List<String> get availableFlagSizes => _flagSizes;

  /// Gets flag URL for a specific country code and size
  static String getFlagUrl(String countryCode, {String size = 'w80'}) {
    final normalizedCode = normalizeCountryCode(countryCode);
    if (!isValidCountryCode(normalizedCode)) {
      throw ArgumentError('Invalid country code: $countryCode');
    }
    if (!_flagSizes.contains(size)) {
      throw ArgumentError('Invalid flag size: $size. Available sizes: $_flagSizes');
    }
    return 'https://flagcdn.com/$size/$normalizedCode.png';
  }

  /// Gets all flag URLs for a country code in different sizes
  static Map<String, String> getAllFlagUrls(String countryCode) {
    final normalizedCode = normalizeCountryCode(countryCode);
    if (!isValidCountryCode(normalizedCode)) {
      throw ArgumentError('Invalid country code: $countryCode');
    }
    
    final Map<String, String> urls = {};
    for (final size in _flagSizes) {
      urls[size] = 'https://flagcdn.com/$size/$normalizedCode.png';
    }
    return urls;
  }
}
