# flutter_app_locker

A Flutter library for locking your app with a PIN code or biometrics (fingerprint/Face ID), similar to App Lock on Android/iOS.

## Platform Support

**This package supports only Android and iOS.**
- Not compatible with web or WASM (WebAssembly) runtimes.
- This is due to the use of native APIs for biometrics and secure storage, which are not available on web platforms.
- See: https://dart.dev/web/wasm

## Features
- Lock your app with a PIN or biometrics
- Supports Android & iOS
- Auto-lock when the app returns after X minutes
- Manual lock/unlock support
- Secure PIN storage using `flutter_secure_storage`
- Simple, easy-to-integrate UI

## Installation

### Add to your pubspec.yaml

If published on pub.dev:
```yaml
dependencies:
  flutter_app_locker: ^0.1.0
```

Set up permissions for `local_auth` as required by its documentation.

## Basic Usage
```dart
import 'package:flutter_app_locker/flutter_app_locker.dart';

void main() {
  runApp(const MyApp());
}

class MyApp extends StatelessWidget {
  const MyApp({super.key});

  @override
  Widget build(BuildContext context) {
    return AppLocker(
      config: const AppLockerConfig(
        timeout: Duration(minutes: 1), 
        enableBiometric: true, 
      ),
      child: MaterialApp(
        home: const HomePage(),
      ),
    );
  }
}
```

## Manual Lock/Unlock
```dart
//lock
AppLocker.lock(context);
//unlock
AppLocker.unlock(context);
```

## Custom Lock Screen UI
You can provide a custom widget via `AppLockerConfig.customLockScreen`.

---

## Documents

### AppLocker Widget
Wrap your entire app with `AppLocker` to enable app locking.

**Constructor:**
```dart
AppLocker({
  required Widget child,
  AppLockerConfig config = const AppLockerConfig(),
})
```
- `child`: Your main app widget (usually MaterialApp).
- `config`: Configuration for lock behavior.

### AppLockerConfig
Configuration for AppLocker.

| Parameter           | Type         | Default                   | Description                                 |
|---------------------|--------------|---------------------------|---------------------------------------------|
| timeout             | Duration     | Duration(minutes: 2)      | Auto-lock timeout after inactivity          |
| enableBiometric     | bool         | true                      | Enable biometric authentication             |
| customLockScreen    | Widget?      | null                      | Custom lock screen widget (optional)        |

**Example:**
```dart
AppLockerConfig(
  timeout: Duration(minutes: 5),
  enableBiometric: false,
  customLockScreen: MyCustomLockScreen(),
)
```

### LockScreen Widget
Default PIN input and biometric lock screen. You can use or customize it as needed.

**Constructor:**
```dart
LockScreen({
  required Function(String pin) onPinEntered,
  bool enableBiometric = false,
  VoidCallback? onBiometricSuccess,
  String? title,
})
```

### Manual Lock/Unlock
- `AppLocker.lock(context)`: Manually lock the app.
- `AppLocker.unlock(context)`: Manually unlock the app.

### PIN Storage
- PIN is securely stored using `flutter_secure_storage`.
- The first time the app is locked, user will be prompted to set a new PIN.

### Biometric Authentication
- Uses `local_auth` for fingerprint/Face ID.
- Enable/disable via `enableBiometric` in `AppLockerConfig`.

### Example: Custom Lock Screen
```dart
AppLocker(
  config: AppLockerConfig(
    customLockScreen: MyCustomLockScreen(),
  ),
  child: MaterialApp(
    home: HomePage(),
  ),
)
```

---

## License
MIT

## Contact
Nguyen Thanh Bien (Nguyễn Thành Biên)  
Email: mortarcloud@gmail.com  
Website: https://algonest.io.vn 