import 'dart:async';
import 'package:flutter/material.dart';
import 'lock_screen.dart';
import 'storage.dart';

/// Configuration for [AppLocker].
/// 
/// This class allows you to customize the behavior of the app locker,
/// including timeout duration, biometric authentication settings, and
/// custom lock screen UI.
class AppLockerConfig {
  /// Auto-lock timeout after inactivity.
  /// 
  /// The app will automatically lock after this duration of inactivity.
  /// Default is 2 minutes.
  final Duration timeout;
  
  /// Enable biometric authentication (fingerprint/Face ID).
  /// 
  /// When enabled, users can authenticate using biometric methods
  /// in addition to PIN authentication.
  /// Default is true.
  final bool enableBiometric;
  
  /// Custom lock screen widget (optional).
  /// 
  /// If provided, this widget will be used instead of the default
  /// lock screen. Useful for customizing the lock screen UI.
  final Widget? customLockScreen;

  /// Creates an [AppLockerConfig].
  /// 
  /// [timeout] specifies how long the app can be inactive before
  /// automatically locking. Default is 2 minutes.
  /// 
  /// [enableBiometric] enables biometric authentication if true.
  /// Default is true.
  /// 
  /// [customLockScreen] allows you to provide a custom lock screen
  /// widget. If null, the default lock screen will be used.
  const AppLockerConfig({
    this.timeout = const Duration(minutes: 2),
    this.enableBiometric = true,
    this.customLockScreen,
  });
}

/// Widget that wraps your app and provides app lock with PIN and biometrics.
/// 
/// This is the main widget that should wrap your app to add locking functionality.
/// It automatically handles app lifecycle changes, timeout management, and
/// authentication flow.
/// 
/// ## Usage
/// 
/// ```dart
/// void main() {
///   runApp(
///     AppLocker(
///       config: AppLockerConfig(
///         timeout: Duration(minutes: 5),
///         enableBiometric: true,
///       ),
///       child: MyApp(),
///     ),
///   );
/// }
/// ```
/// 
/// ## Features
/// 
/// * Automatic app locking after timeout
/// * Background app detection
/// * PIN and biometric authentication
/// * Customizable lock screen
/// * Manual lock/unlock methods
class AppLocker extends StatefulWidget {
  /// The main app widget (usually MaterialApp).
  /// 
  /// This is the widget that will be displayed when the app is unlocked.
  /// It should typically be your main app widget.
  final Widget child;
  
  /// Configuration for lock behavior.
  /// 
  /// Use this to customize timeout duration, biometric settings,
  /// and lock screen appearance.
  final AppLockerConfig config;

  /// Creates an [AppLocker] widget.
  /// 
  /// [child] is the main app widget to be displayed when unlocked.
  /// [config] provides configuration options for the lock behavior.
  /// If not provided, default configuration will be used.
  const AppLocker({super.key, required this.child, this.config = const AppLockerConfig()});

  /// Returns the nearest [_AppLockerState] up the widget tree.
  /// 
  /// This method is used internally to access the AppLocker state
  /// from anywhere in the widget tree.
  static _AppLockerState? of(BuildContext context) => context.findAncestorStateOfType<_AppLockerState>();

  /// Manually lock the app.
  /// 
  /// Call this method to immediately lock the app, regardless of
  /// the current timeout state.
  /// 
  /// ```dart
  /// AppLocker.lock(context);
  /// ```
  static void lock(BuildContext context) => of(context)?._lock();
  
  /// Manually unlock the app.
  /// 
  /// Call this method to immediately unlock the app and reset
  /// the timeout timer.
  /// 
  /// ```dart
  /// AppLocker.unlock(context);
  /// ```
  static void unlock(BuildContext context) => of(context)?._unlock();

  @override
  State<AppLocker> createState() => _AppLockerState();
}

class _AppLockerState extends State<AppLocker> with WidgetsBindingObserver {
  bool _locked = false;
  Timer? _timer;
  DateTime? _backgroundTime;
  String? _pin;

  @override
  void initState() {
    super.initState();
    WidgetsBinding.instance.addObserver(this);
    _initPin();
    _lock();
  }

  @override
  void dispose() {
    WidgetsBinding.instance.removeObserver(this);
    _timer?.cancel();
    super.dispose();
  }

  Future<void> _initPin() async {
    _pin = await Storage.getPin();
    setState(() {});
  }

  void _lock() {
    setState(() {
      _locked = true;
    });
  }

  void _unlock() {
    setState(() {
      _locked = false;
    });
    _resetTimeout();
  }

  void _resetTimeout() {
    _timer?.cancel();
    _timer = Timer(widget.config.timeout, _lock);
  }

  @override
  void didChangeAppLifecycleState(AppLifecycleState state) {
    if (state == AppLifecycleState.paused) {
      _backgroundTime = DateTime.now();
    } else if (state == AppLifecycleState.resumed) {
      if (_backgroundTime != null) {
        final diff = DateTime.now().difference(_backgroundTime!);
        if (diff >= widget.config.timeout) {
          _lock();
        }
      }
    }
  }

  Future<void> _onPinEntered(String pin) async {
    if (_pin == null) {
      await Storage.savePin(pin);
      setState(() {
        _pin = pin;
        _locked = false;
      });
      _resetTimeout();
    } else if (pin == _pin) {
      setState(() {
        _locked = false;
      });
      _resetTimeout();
    } else {
    }
  }

  Future<void> _onBiometricSuccess() async {
    setState(() {
      _locked = false;
    });
    _resetTimeout();
  }

  @override
  Widget build(BuildContext context) {
    if (_locked) {
      if (widget.config.customLockScreen != null) {
        return widget.config.customLockScreen!;
      }
      return LockScreen(
        onPinEntered: _onPinEntered,
        enableBiometric: widget.config.enableBiometric,
        onBiometricSuccess: _onBiometricSuccess,
        title: _pin == null ? 'Set a new PIN' : null,
      );
    }
    return GestureDetector(
      onTap: _resetTimeout,
      behavior: HitTestBehavior.translucent,
      child: widget.child,
    );
  }
} 