import 'package:local_auth/local_auth.dart';

/// A helper class for biometric authentication operations.
/// 
/// This class provides methods to check biometric availability and
/// perform biometric authentication using the local_auth package.
/// It supports fingerprint and Face ID authentication on supported devices.
/// 
/// ## Supported Platforms
/// 
/// * iOS: Touch ID and Face ID
/// * Android: Fingerprint authentication
/// * Web: Not supported (returns false)
/// 
/// ## Usage
/// 
/// ```dart
/// // Check if biometric authentication is available
/// bool available = await BiometricHelper.isBiometricAvailable();
/// 
/// if (available) {
///   // Perform biometric authentication
///   bool success = await BiometricHelper.authenticate(
///     reason: 'Unlock your app'
///   );
///   
///   if (success) {
///     // Authentication successful
///   }
/// }
/// ```
class BiometricHelper {
  static final _auth = LocalAuthentication();

  /// Checks if biometric authentication is available on the device.
  /// 
  /// This method checks both if the device supports biometric authentication
  /// and if biometric authentication is available for use.
  /// 
  /// Returns true if biometric authentication is available and can be used.
  /// 
  /// ```dart
  /// bool available = await BiometricHelper.isBiometricAvailable();
  /// if (available) {
  ///   // Enable biometric authentication in your UI
  /// }
  /// ```
  static Future<bool> isBiometricAvailable() async {
    return await _auth.canCheckBiometrics && await _auth.isDeviceSupported();
  }

  /// Performs biometric authentication.
  /// 
  /// This method prompts the user to authenticate using biometric methods
  /// (fingerprint/Face ID). The authentication process is handled by the
  /// platform's native biometric authentication system.
  /// 
  /// [reason] is the message displayed to the user during authentication.
  /// Default is 'Authenticate to unlock the app'.
  /// 
  /// Returns true if authentication is successful, false otherwise.
  /// 
  /// ```dart
  /// bool success = await BiometricHelper.authenticate(
  ///   reason: 'Please authenticate to access your data'
  /// );
  /// 
  /// if (success) {
  ///   // User successfully authenticated
  ///   unlockApp();
  /// }
  /// ```
  static Future<bool> authenticate({String reason = 'Authenticate to unlock the app'}) async {
    try {
      return await _auth.authenticate(
        localizedReason: reason,
        options: const AuthenticationOptions(
          biometricOnly: true,
          stickyAuth: true,
        ),
      );
    } catch (e) {
      return false;
    }
  }
} 