import 'package:flutter/material.dart';
import 'biometric_helper.dart';

/// A customizable lock screen widget for PIN and biometric authentication.
/// 
/// This widget provides a complete lock screen interface with:
/// * PIN input with visual feedback
/// * Biometric authentication button
/// * Customizable title and appearance
/// * Error handling and validation
/// 
/// ## Usage
/// 
/// ```dart
/// LockScreen(
///   onPinEntered: (pin) {
///     // Handle PIN validation
///     if (pin == correctPin) {
///       // Unlock app
///     }
///   },
///   enableBiometric: true,
///   onBiometricSuccess: () {
///     // Handle successful biometric authentication
///   },
///   title: 'Enter your PIN',
/// )
/// ```
class LockScreen extends StatefulWidget {
  /// Callback function called when a PIN is entered.
  /// 
  /// This function receives the entered PIN as a parameter.
  /// It should handle PIN validation and app unlocking logic.
  final Function(String pin) onPinEntered;
  
  /// Whether to enable biometric authentication.
  /// 
  /// When true, a fingerprint icon will be displayed
  /// that users can tap to authenticate with biometrics.
  /// Default is false.
  final bool enableBiometric;
  
  /// Callback function called when biometric authentication succeeds.
  /// 
  /// This function is called when the user successfully
  /// authenticates using biometric methods (fingerprint/Face ID).
  final VoidCallback? onBiometricSuccess;
  
  /// Custom title text for the lock screen.
  /// 
  /// If not provided, a default title will be used.
  /// Useful for providing context-specific messages.
  final String? title;

  /// Creates a [LockScreen] widget.
  /// 
  /// [onPinEntered] is required and handles PIN validation.
  /// [enableBiometric] enables biometric authentication if true.
  /// [onBiometricSuccess] is called when biometric auth succeeds.
  /// [title] provides custom title text for the lock screen.
  const LockScreen({
    super.key,
    required this.onPinEntered,
    this.enableBiometric = false,
    this.onBiometricSuccess,
    this.title,
  });

  @override
  State<LockScreen> createState() => _LockScreenState();
}

class _LockScreenState extends State<LockScreen> {
  String _pin = '';
  String? _error;

  @override
  void initState() {
    super.initState();
    if (widget.enableBiometric) {
      _tryBiometric();
    }
  }

  Future<void> _tryBiometric() async {
    if (await BiometricHelper.isBiometricAvailable()) {
      final success = await BiometricHelper.authenticate();
      if (success) {
        widget.onBiometricSuccess?.call();
      }
    }
  }

  void _onKeyTap(String value) {
    setState(() {
      if (_pin.length < 6) {
        _pin += value;
        _error = null;
      }
      if (_pin.length == 6) {
        widget.onPinEntered(_pin);
      }
    });
  }

  void _onBackspace() {
    setState(() {
      if (_pin.isNotEmpty) {
        _pin = _pin.substring(0, _pin.length - 1);
      }
    });
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      body: SafeArea(
        child: Center(
          child: Column(
            mainAxisAlignment: MainAxisAlignment.center,
            children: [
              Text(widget.title ?? 'Enter PIN to unlock', style: const TextStyle(fontSize: 20)),
              const SizedBox(height: 24),
              Row(
                mainAxisAlignment: MainAxisAlignment.center,
                children: List.generate(6, (i) => _buildPinDot(i < _pin.length)),
              ),
              if (_error != null) ...[
                const SizedBox(height: 12),
                Text(_error!, style: const TextStyle(color: Colors.red)),
              ],
              const SizedBox(height: 24),
              _buildNumPad(),
              if (widget.enableBiometric)
                IconButton(
                  icon: const Icon(Icons.fingerprint, size: 36),
                  onPressed: _tryBiometric,
                ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildPinDot(bool filled) {
    return Container(
      margin: const EdgeInsets.symmetric(horizontal: 8),
      width: 16,
      height: 16,
      decoration: BoxDecoration(
        color: filled ? Colors.blue : Colors.transparent,
        border: Border.all(color: Colors.blue),
        shape: BoxShape.circle,
      ),
    );
  }

  Widget _buildNumPad() {
    return Column(
      children: [
        for (var row in [
          ['1', '2', '3'],
          ['4', '5', '6'],
          ['7', '8', '9'],
          ['', '0', '<'],
        ])
          Row(
            mainAxisAlignment: MainAxisAlignment.center,
            children: row.map((v) {
              if (v == '<') {
                return _buildNumButton(const Icon(Icons.backspace), _onBackspace);
              } else if (v.isEmpty) {
                return const SizedBox(width: 64);
              } else {
                return _buildNumButton(Text(v, style: const TextStyle(fontSize: 24)), () => _onKeyTap(v));
              }
            }).toList(),
          ),
      ],
    );
  }

  Widget _buildNumButton(Widget child, VoidCallback onTap) {
    return Padding(
      padding: const EdgeInsets.all(8.0),
      child: InkWell(
        onTap: onTap,
        borderRadius: BorderRadius.circular(32),
        child: Container(
          width: 64,
          height: 64,
          alignment: Alignment.center,
          decoration: BoxDecoration(
            border: Border.all(color: Colors.blue),
            borderRadius: BorderRadius.circular(32),
          ),
          child: child,
        ),
      ),
    );
  }
} 