import 'package:flutter_secure_storage/flutter_secure_storage.dart';

/// A secure storage utility for managing PIN codes.
/// 
/// This class provides methods to securely store, retrieve, and delete
/// PIN codes using flutter_secure_storage. All operations are asynchronous
/// and use platform-specific secure storage mechanisms.
/// 
/// ## Security
/// 
/// PIN codes are stored using flutter_secure_storage, which provides:
/// * iOS: Keychain storage
/// * Android: Encrypted SharedPreferences
/// * Web: Encrypted localStorage
/// 
/// ## Usage
/// 
/// ```dart
/// // Save a PIN
/// await Storage.savePin('123456');
/// 
/// // Retrieve the PIN
/// String? pin = await Storage.getPin();
/// 
/// // Delete the PIN
/// await Storage.deletePin();
/// ```
class Storage {
  static const _pinKey = 'app_locker_pin';
  static final _storage = FlutterSecureStorage();

  /// Saves a PIN code to secure storage.
  /// 
  /// The PIN is stored securely using platform-specific encryption.
  /// This method is asynchronous and should be awaited.
  /// 
  /// [pin] is the PIN code to store.
  /// 
  /// ```dart
  /// await Storage.savePin('123456');
  /// ```
  static Future<void> savePin(String pin) async {
    await _storage.write(key: _pinKey, value: pin);
  }

  /// Retrieves the stored PIN code from secure storage.
  /// 
  /// Returns the stored PIN code, or null if no PIN is stored.
  /// This method is asynchronous and should be awaited.
  /// 
  /// ```dart
  /// String? pin = await Storage.getPin();
  /// if (pin != null) {
  ///   // PIN exists, validate it
  /// }
  /// ```
  static Future<String?> getPin() async {
    return await _storage.read(key: _pinKey);
  }

  /// Deletes the stored PIN code from secure storage.
  /// 
  /// This method removes the PIN code from secure storage.
  /// After calling this method, [getPin()] will return null.
  /// This method is asynchronous and should be awaited.
  /// 
  /// ```dart
  /// await Storage.deletePin();
  /// ```
  static Future<void> deletePin() async {
    await _storage.delete(key: _pinKey);
  }
} 