// flutter_permission_guard.dart
// Thư viện kiểm soát quyền truy cập widget bằng permission_handler

import 'package:flutter/material.dart';
import 'package:permission_handler/permission_handler.dart';

/// Widget kiểm tra và xin quyền trước khi hiển thị widget con.
///
/// Sử dụng cho camera, microphone, location, v.v.
class PermissionGuard extends StatefulWidget {
  /// Danh sách quyền cần kiểm tra (dùng cho PermissionGuard.multi)
  final List<Permission> permissions;

  /// Widget con sẽ hiển thị khi đã đủ quyền
  final Widget child;

  /// Widget hiển thị khi bị từ chối quyền (có thể custom)
  final Widget? deniedWidget;

  /// Callback khi bị từ chối quyền (không bắt buộc)
  final VoidCallback? onDenied;

  /// Tạo PermissionGuard cho 1 quyền
  PermissionGuard({
    super.key,
    required Permission permission,
    required this.child,
    this.deniedWidget,
    this.onDenied,
  }) : permissions = [permission];

  /// Tạo PermissionGuard cho nhiều quyền cùng lúc
  const PermissionGuard.multi({
    super.key,
    required this.permissions,
    required this.child,
    this.deniedWidget,
    this.onDenied,
  });

  @override
  State<PermissionGuard> createState() => _PermissionGuardState();
}

class _PermissionGuardState extends State<PermissionGuard> {
  bool _checking = true;
  bool _denied = false;
  bool _permanentlyDenied = false;

  @override
  void initState() {
    super.initState();
    _checkPermissions();
  }

  Future<void> _checkPermissions() async {
    setState(() {
      _checking = true;
      _denied = false;
      _permanentlyDenied = false;
    });
    bool allGranted = true;
    for (final permission in widget.permissions) {
      final status = await permission.status;
      if (status.isGranted) continue;
      allGranted = false;
    }
    if (allGranted) {
      setState(() {
        _checking = false;
        _denied = false;
        _permanentlyDenied = false;
      });
      return;
    }
    // Yêu cầu quyền
    final statuses = await widget.permissions.request();
    bool granted = true;
    bool permanentlyDenied = false;
    for (final p in widget.permissions) {
      final s = statuses[p];
      if (s == null || !s.isGranted) granted = false;
      if (s != null && s.isPermanentlyDenied) permanentlyDenied = true;
    }
    setState(() {
      _checking = false;
      _denied = !granted;
      _permanentlyDenied = permanentlyDenied;
    });
    if (!granted && widget.onDenied != null) widget.onDenied!();
  }

  @override
  Widget build(BuildContext context) {
    if (_checking) {
      return const Center(child: CircularProgressIndicator());
    }
    if (!_denied) {
      return widget.child;
    }
    // Nếu bị từ chối quyền
    return widget.deniedWidget ??
        _DefaultDeniedWidget(
          permanentlyDenied: _permanentlyDenied,
          onOpenSettings: () async {
            await openAppSettings();
          },
          onRetry: _checkPermissions,
        );
  }
}

/// Widget mặc định khi bị từ chối quyền
class _DefaultDeniedWidget extends StatelessWidget {
  final bool permanentlyDenied;
  final VoidCallback onOpenSettings;
  final VoidCallback onRetry;

  const _DefaultDeniedWidget({
    required this.permanentlyDenied,
    required this.onOpenSettings,
    required this.onRetry,
  });

  @override
  Widget build(BuildContext context) {
    return Center(
      child: Padding(
        padding: const EdgeInsets.all(24.0),
        child: Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            Icon(Icons.lock_outline, size: 48, color: Colors.grey[600]),
            const SizedBox(height: 16),
            Text(
              permanentlyDenied
                  ? 'Quyền đã bị từ chối vĩnh viễn.'
                  : 'Bạn cần cấp quyền để tiếp tục.',
              textAlign: TextAlign.center,
              style: Theme.of(context).textTheme.titleMedium,
            ),
            const SizedBox(height: 16),
            if (permanentlyDenied)
              ElevatedButton.icon(
                icon: const Icon(Icons.settings),
                label: const Text('Mở cài đặt'),
                onPressed: onOpenSettings,
              )
            else
              ElevatedButton.icon(
                icon: const Icon(Icons.refresh),
                label: const Text('Thử lại'),
                onPressed: onRetry,
              ),
          ],
        ),
      ),
    );
  }
}
