import 'package:flutter/material.dart';
import 'package:lan_discover/lan_discover.dart';

void main() {
  runApp(const LanDiscoverDemoApp());
}

class LanDiscoverDemoApp extends StatelessWidget {
  const LanDiscoverDemoApp({super.key});

  @override
  Widget build(BuildContext context) {
    return MaterialApp(
      title: 'LAN Discover Demo',
      theme: ThemeData(
        colorScheme: ColorScheme.fromSeed(seedColor: Colors.blue),
      ),
      home: const LanDiscoverHomePage(),
    );
  }
}

class LanDiscoverHomePage extends StatefulWidget {
  const LanDiscoverHomePage({super.key});

  @override
  State<LanDiscoverHomePage> createState() => _LanDiscoverHomePageState();
}

class _LanDiscoverHomePageState extends State<LanDiscoverHomePage> {
  List<LanDevice> _devices = [];
  bool _scanning = false;
  double _progress = 0;
  String? _error;

  Future<void> _scan() async {
    setState(() {
      _scanning = true;
      _devices = [];
      _progress = 0;
      _error = null;
    });
    try {
      final scanner = LanScanner();
      final devices = await scanner.scan(
        onProgress: (progress) {
          setState(() {
            _progress = progress;
          });
        },
      );
      setState(() {
        _devices = devices;
      });
    } catch (e) {
      setState(() {
        _error = e.toString();
      });
    } finally {
      setState(() {
        _scanning = false;
      });
    }
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: const Text('LAN Discover Demo'),
      ),
      body: Padding(
        padding: const EdgeInsets.all(16.0),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.stretch,
          children: [
            ElevatedButton(
              onPressed: _scanning ? null : _scan,
              child: const Text('Scan LAN Devices'),
            ),
            if (_scanning)
              Padding(
                padding: const EdgeInsets.symmetric(vertical: 16.0),
                child: LinearProgressIndicator(value: _progress),
              ),
            if (_error != null)
              Padding(
                padding: const EdgeInsets.symmetric(vertical: 8.0),
                child: Text('Error: \\$_error', style: const TextStyle(color: Colors.red)),
              ),
            Expanded(
              child: _devices.isEmpty
                  ? const Center(child: Text('No devices found.'))
                  : ListView.builder(
                      itemCount: _devices.length,
                      itemBuilder: (context, index) {
                        final device = _devices[index];
                        return ListTile(
                          leading: const Icon(Icons.devices),
                          title: Text(device.ip),
                          subtitle: Text(device.hostname ?? device.mdnsName ?? 'Unknown'),
                          trailing: device.openPorts != null && device.openPorts!.isNotEmpty
                              ? Text('Ports: \\${device.openPorts!.join(", ")}')
                              : null,
                        );
                      },
                    ),
            ),
          ],
        ),
      ),
    );
  }
}
