import 'dart:async';
import 'dart:io';
import 'package:network_info_plus/network_info_plus.dart';
import 'package:ping_discover_network_forked/ping_discover_network_forked.dart';
import 'device_model.dart';

class LanScanner {
  /// Scans the local network for devices.
  ///
  /// [ports]: If provided, will check if these ports are open on each device.
  /// [timeout]: Timeout for each ping.
  /// [onProgress]: Callback for progress (0.0 - 1.0).
  /// [resolveHostnames]: If true, will try to resolve hostnames.
  /// [isCancelled]: If provided, will check this function to support cancellation.
  Future<List<LanDevice>> scan({
    List<int>? ports,
    Duration timeout = const Duration(milliseconds: 500),
    void Function(double progress)? onProgress,
    bool resolveHostnames = true,
    bool Function()? isCancelled,
  }) async {
    final info = NetworkInfo();
    final wifiIP = await info.getWifiIP();
    final wifiSubmask = await info.getWifiSubmask();
    if (wifiIP == null || wifiSubmask == null) {
      throw Exception('Could not get local IP or subnet mask. Are you connected to Wi-Fi?');
    }

    // Calculate subnet
    final subnet = _getSubnet(wifiIP, wifiSubmask);
    final base = subnet.split('.').take(3).join('.');
    final List<LanDevice> foundDevices = [];
    final int total = 254;
    int current = 0;
    final List<Future<void>> futures = [];

    for (int i = 1; i <= 254; i++) {
      if (isCancelled?.call() == true) break;
      final ip = '$base.$i';
      futures.add(_scanIp(ip, ports, timeout, resolveHostnames).then((device) {
        if (device != null) {
          foundDevices.add(device);
        }
        current++;
        if (onProgress != null) {
          onProgress(current / total);
        }
      }));
    }
    await Future.wait(futures);
    return foundDevices;
  }

  String _getSubnet(String ip, String mask) {
    // Simple IPv4 subnet calculation
    final ipParts = ip.split('.').map(int.parse).toList();
    final maskParts = mask.split('.').map(int.parse).toList();
    final subnetParts = List.generate(4, (i) => ipParts[i] & maskParts[i]);
    return subnetParts.join('.');
  }

  Future<LanDevice?> _scanIp(
    String ip,
    List<int>? ports,
    Duration timeout,
    bool resolveHostnames,
  ) async {
    try {
      final isReachable = await _ping(ip, timeout);
      if (!isReachable) return null;
      String? hostname;
      if (resolveHostnames) {
        try {
          final names = await InternetAddress(ip).reverse();
          hostname = names.host;
        } catch (_) {}
      }
      List<int>? openPorts;
      if (ports != null && ports.isNotEmpty) {
        openPorts = [];
        for (final port in ports) {
          final open = await _checkPort(ip, port, timeout);
          if (open) openPorts.add(port);
        }
      }
      return LanDevice(ip: ip, hostname: hostname, openPorts: openPorts);
    } catch (_) {
      return null;
    }
  }

  Future<bool> _ping(String ip, Duration timeout) async {
    try {
      final stream = NetworkAnalyzer.discover2(ip, 1, timeout: timeout);
      await for (final addr in stream) {
        if (addr.exists) return true;
      }
      return false;
    } catch (_) {
      return false;
    }
  }

  Future<bool> _checkPort(String ip, int port, Duration timeout) async {
    try {
      final socket = await Socket.connect(ip, port, timeout: timeout);
      socket.destroy();
      return true;
    } catch (_) {
      return false;
    }
  }
} 