import 'dart:io';
import 'package:multicast_dns/multicast_dns.dart';
import 'device_model.dart';

class MdnsDiscovery {
  Future<List<LanDevice>> discover({
    void Function(double progress)? onProgress,
    bool Function()? isCancelled,
  }) async {
    final List<LanDevice> devices = [];
    final MDnsClient client = MDnsClient(rawDatagramSocketFactory: (dynamic host, int port, {bool? reuseAddress, bool? reusePort, int? ttl}) {
      return RawDatagramSocket.bind(host, port, reuseAddress: true, reusePort: false, ttl: ttl ?? 1);
    });
    await client.start();
    try {
      // Query for all services
      await for (final PtrResourceRecord ptr in client.lookup<PtrResourceRecord>(ResourceRecordQuery.serverPointer('_services._dns-sd._udp.local'))) {
        if (isCancelled?.call() == true) break;
        // For each service, query for instances
        await for (final PtrResourceRecord _ in client.lookup<PtrResourceRecord>(ResourceRecordQuery.serverPointer(ptr.domainName))) {
          if (isCancelled?.call() == true) break;
          // Query for SRV and TXT records
          String? host;
          Map<String, String> txtData = {};
          await for (final SrvResourceRecord srv in client.lookup<SrvResourceRecord>(ResourceRecordQuery.service(ptr.domainName))) {
            host = srv.target;
          }
          await for (final TxtResourceRecord txt in client.lookup<TxtResourceRecord>(ResourceRecordQuery.text(ptr.domainName))) {
            final dynamic raw = txt.text;
            final Iterable<String> txtList =
              raw is Iterable ? raw.cast<String>() :
              raw is String ? [raw] :
              const <String>[];
            for (final data in txtList) {
              if (data.contains('=')) {
                final parts = data.split('=');
                if (parts.length == 2) {
                  txtData[parts[0]] = parts[1];
                }
              }
            }
          }
          // Query for IP address
          String? ip;
          if (host != null) {
            await for (final IPAddressResourceRecord addr in client.lookup<IPAddressResourceRecord>(ResourceRecordQuery.addressIPv4(host))) {
              ip = addr.address.address;
            }
          }
          if (ip != null) {
            devices.add(LanDevice(
              ip: ip,
              mdnsName: ptr.domainName,
              mdnsInfo: txtData.isNotEmpty ? txtData : null,
            ));
          }
        }
      }
    } finally {
      client.stop();
    }
    return devices;
  }
} 