import 'package:flutter/material.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:netcache_image/netcache_image.dart';

void main() {
  group('NetCacheImage Widget Tests', () {
    testWidgets('should display placeholder while loading',
        (WidgetTester tester) async {
      const placeholder = Icon(Icons.image);

      await tester.pumpWidget(
        MaterialApp(
          home: Scaffold(
            body: NetCacheImage(
              url: 'https://example.com/test.jpg',
              placeholder: placeholder,
            ),
          ),
        ),
      );

      expect(find.byIcon(Icons.image), findsOneWidget);
    });

    testWidgets('should handle null placeholder and error widget',
        (WidgetTester tester) async {
      await tester.pumpWidget(
        MaterialApp(
          home: Scaffold(
            body: NetCacheImage(
              url: 'https://example.com/test.jpg',
            ),
          ),
        ),
      );

      // Should not throw any errors
      expect(tester.takeException(), isNull);
    });

    testWidgets('should rebuild when URL changes', (WidgetTester tester) async {
      await tester.pumpWidget(
        MaterialApp(
          home: Scaffold(
            body: NetCacheImage(
              url: 'https://example.com/test1.jpg',
            ),
          ),
        ),
      );

      await tester.pumpWidget(
        MaterialApp(
          home: Scaffold(
            body: NetCacheImage(
              url: 'https://example.com/test2.jpg',
            ),
          ),
        ),
      );

      // Should not throw any errors
      expect(tester.takeException(), isNull);
    });

    testWidgets('should show placeholder initially',
        (WidgetTester tester) async {
      const placeholder = Icon(Icons.image);

      await tester.pumpWidget(
        MaterialApp(
          home: Scaffold(
            body: NetCacheImage(
              url: 'https://example.com/test.jpg',
              placeholder: placeholder,
            ),
          ),
        ),
      );

      // Should show placeholder immediately
      expect(find.byIcon(Icons.image), findsOneWidget);
    });

    testWidgets('should handle custom configuration',
        (WidgetTester tester) async {
      const placeholder = Icon(Icons.image);
      const errorWidget = Icon(Icons.error);

      await tester.pumpWidget(
        MaterialApp(
          home: Scaffold(
            body: NetCacheImage(
              url: 'https://example.com/test.jpg',
              placeholder: placeholder,
              errorWidget: errorWidget,
              cacheExpiration: const Duration(days: 30),
              maxRetries: 5,
              retryDelay: const Duration(seconds: 1),
            ),
          ),
        ),
      );

      // Should show placeholder initially
      expect(find.byIcon(Icons.image), findsOneWidget);
    });
  });

  group('NetCacheImage Static Methods Tests', () {
    test('preload should not throw with empty list', () async {
      expect(() async {
        await NetCacheImage.preload([]);
      }, returnsNormally);
    });

    test('preload should not throw with valid URLs', () async {
      expect(() async {
        await NetCacheImage.preload([
          'https://picsum.photos/300/200?random=1',
          'https://picsum.photos/300/200?random=2',
        ]);
      }, returnsNormally);
    });

    test('clearCache should not throw', () async {
      expect(() async {
        await NetCacheImage.clearCache();
      }, returnsNormally);
    });

    test('clearExpiredCache should not throw', () async {
      expect(() async {
        await NetCacheImage.clearExpiredCache();
      }, returnsNormally);
    });

    test('getCacheSize should return non-negative value', () async {
      final size = await NetCacheImage.getCacheSize();
      expect(size, isA<int>());
      expect(size, greaterThanOrEqualTo(0));
    });
  });

  group('NetCacheImage Constructor Tests', () {
    test('should create widget with required parameters', () {
      const widget = NetCacheImage(
        url: 'https://example.com/test.jpg',
      );

      expect(widget.url, 'https://example.com/test.jpg');
      expect(widget.placeholder, isNull);
      expect(widget.errorWidget, isNull);
      expect(widget.cacheExpiration, isNull);
      expect(widget.maxRetries, 3);
      expect(widget.retryDelay, const Duration(seconds: 2));
    });

    test('should create widget with all parameters', () {
      const placeholder = Icon(Icons.image);
      const errorWidget = Icon(Icons.error);
      const cacheExpiration = Duration(days: 7);
      const maxRetries = 5;
      const retryDelay = Duration(seconds: 1);

      const widget = NetCacheImage(
        url: 'https://example.com/test.jpg',
        placeholder: placeholder,
        errorWidget: errorWidget,
        cacheExpiration: cacheExpiration,
        maxRetries: maxRetries,
        retryDelay: retryDelay,
      );

      expect(widget.url, 'https://example.com/test.jpg');
      expect(widget.placeholder, placeholder);
      expect(widget.errorWidget, errorWidget);
      expect(widget.cacheExpiration, cacheExpiration);
      expect(widget.maxRetries, maxRetries);
      expect(widget.retryDelay, retryDelay);
    });
  });
}
