import 'dart:async';

import 'package:flutter/material.dart';
import 'package:offline_messenger/offline_messenger.dart';

void main() {
  runApp(const OfflineMessengerExampleApp());
}

/// Example application demonstrating the offline_messenger package functionality.
class OfflineMessengerExampleApp extends StatelessWidget {
  const OfflineMessengerExampleApp({super.key});

  @override
  Widget build(BuildContext context) {
    return MaterialApp(
      title: 'OfflineMessenger Example',
      theme: ThemeData(
        colorScheme: ColorScheme.fromSeed(seedColor: Colors.blue),
        useMaterial3: true,
      ),
      home: const OfflineMessengerScreen(),
    );
  }
}

/// Main screen for the OfflineMessenger example application.
class OfflineMessengerScreen extends StatefulWidget {
  const OfflineMessengerScreen({super.key});

  @override
  State<OfflineMessengerScreen> createState() => _OfflineMessengerScreenState();
}

class _OfflineMessengerScreenState extends State<OfflineMessengerScreen> {
  final OfflineMessenger _offlineMessenger = OfflineMessenger();
  final TextEditingController _messageController = TextEditingController();
  final List<OfflineMessage> _messages = [];
  final List<OfflineDevice> _discoveredDevices = [];
  OfflineConnectionState _connectionState = OfflineConnectionState.unavailable;
  OfflineDevice? _connectedDevice;
  bool _isInitialized = false;

  @override
  void initState() {
    super.initState();
    _initializeOfflineMessenger();
  }

  @override
  void dispose() {
    _offlineMessenger.dispose();
    _messageController.dispose();
    super.dispose();
  }

  /// Initializes the OfflineMessenger instance.
  Future<void> _initializeOfflineMessenger() async {
    final success = await _offlineMessenger.initialize();
    if (success) {
      setState(() {
        _isInitialized = true;
      });

      // Listen to connection state changes
      _offlineMessenger.connectionStateStream.listen((state) {
        setState(() {
          _connectionState = state;
        });
      });

      // Listen to discovered devices
      _offlineMessenger.discoveredDevicesStream.listen((devices) {
        setState(() {
          _discoveredDevices.clear();
          _discoveredDevices.addAll(devices);
        });
      });

      // Listen to messages
      _offlineMessenger.messagesStream.listen((message) {
        setState(() {
          _messages.add(message);
        });
      });
    } else {
      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(
            content: Text('Failed to initialize Bluetooth. Please check permissions and Bluetooth settings.'),
            backgroundColor: Colors.red,
          ),
        );
      }
    }
  }

  /// Starts scanning for nearby devices.
  Future<void> _startScanning() async {
    await _offlineMessenger.startScanning();
  }

  /// Connects to a specific device.
  Future<void> _connectToDevice(OfflineDevice device) async {
    final success = await _offlineMessenger.connectToDevice(device);
    if (success) {
      setState(() {
        _connectedDevice = device;
      });
      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(
            content: Text('Connected to ${device.name}'),
            backgroundColor: Colors.green,
          ),
        );
      }
    } else {
      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(
            content: Text('Failed to connect to device'),
            backgroundColor: Colors.red,
          ),
        );
      }
    }
  }

  /// Disconnects from the current device.
  Future<void> _disconnect() async {
    await _offlineMessenger.disconnect();
    setState(() {
      _connectedDevice = null;
    });
  }

  /// Sends a message to the connected device.
  Future<void> _sendMessage() async {
    final message = _messageController.text.trim();
    if (message.isEmpty) return;

    final success = await _offlineMessenger.sendMessage(message);
    if (success) {
      _messageController.clear();
    } else {
      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(
            content: Text('Failed to send message'),
            backgroundColor: Colors.red,
          ),
        );
      }
    }
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: const Text('OfflineMessenger Example'),
        backgroundColor: Theme.of(context).colorScheme.inversePrimary,
        actions: [
          IconButton(
            icon: Icon(_offlineMessenger.isScanning ? Icons.stop : Icons.search),
            onPressed: _offlineMessenger.isScanning ? _offlineMessenger.stopScanning : _startScanning,
            tooltip: _offlineMessenger.isScanning ? 'Stop Scanning' : 'Start Scanning',
          ),
        ],
      ),
      body: !_isInitialized
          ? const Center(
              child: Column(
                mainAxisAlignment: MainAxisAlignment.center,
                children: [
                  CircularProgressIndicator(),
                  SizedBox(height: 16),
                  Text('Initializing Bluetooth...'),
                ],
              ),
            )
          : Column(
              children: [
                // Connection status
                Container(
                  padding: const EdgeInsets.all(16),
                  color: _getStatusColor(),
                  child: Row(
                    children: [
                      Icon(_getStatusIcon(), color: Colors.white),
                      const SizedBox(width: 8),
                      Expanded(
                        child: Text(
                          _connectionState.description,
                          style: const TextStyle(color: Colors.white, fontWeight: FontWeight.bold),
                        ),
                      ),
                      if (_connectedDevice != null)
                        IconButton(
                          icon: const Icon(Icons.close, color: Colors.white),
                          onPressed: _disconnect,
                          tooltip: 'Disconnect',
                        ),
                    ],
                  ),
                ),

                // Discovered devices
                if (_discoveredDevices.isNotEmpty)
                  Container(
                    height: 120,
                    padding: const EdgeInsets.all(8),
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        const Text('Discovered Devices:', style: TextStyle(fontWeight: FontWeight.bold)),
                        const SizedBox(height: 8),
                        Expanded(
                          child: ListView.builder(
                            scrollDirection: Axis.horizontal,
                            itemCount: _discoveredDevices.length,
                            itemBuilder: (context, index) {
                              final device = _discoveredDevices[index];
                              return Card(
                                margin: const EdgeInsets.only(right: 8),
                                child: Padding(
                                  padding: const EdgeInsets.all(8),
                                  child: Column(
                                    mainAxisSize: MainAxisSize.min,
                                    children: [
                                      Text(
                                        device.name,
                                        style: const TextStyle(fontWeight: FontWeight.bold),
                                      ),
                                      Text('RSSI: ${device.rssi}'),
                                      ElevatedButton(
                                        onPressed: () => _connectToDevice(device),
                                        child: const Text('Connect'),
                                      ),
                                    ],
                                  ),
                                ),
                              );
                            },
                          ),
                        ),
                      ],
                    ),
                  ),

                // Messages
                Expanded(
                  child: _messages.isEmpty
                      ? const Center(
                          child: Text(
                            'No messages yet.\nConnect to a device to start chatting!',
                            textAlign: TextAlign.center,
                            style: TextStyle(color: Colors.grey),
                          ),
                        )
                      : ListView.builder(
                          padding: const EdgeInsets.all(8),
                          itemCount: _messages.length,
                          itemBuilder: (context, index) {
                            final message = _messages[index];
                            return _buildMessageWidget(message);
                          },
                        ),
                ),

                // Message input
                if (_connectedDevice != null)
                  Container(
                    padding: const EdgeInsets.all(8),
                    child: Row(
                      children: [
                        Expanded(
                          child: TextField(
                            controller: _messageController,
                            decoration: const InputDecoration(
                              hintText: 'Type a message...',
                              border: OutlineInputBorder(),
                            ),
                            onSubmitted: (_) => _sendMessage(),
                          ),
                        ),
                        const SizedBox(width: 8),
                        IconButton(
                          onPressed: _sendMessage,
                          icon: const Icon(Icons.send),
                          tooltip: 'Send Message',
                        ),
                      ],
                    ),
                  ),
              ],
            ),
    );
  }

  /// Builds a message widget.
  Widget _buildMessageWidget(OfflineMessage message) {
    final isFromMe = message.isFromMe;
    return Align(
      alignment: isFromMe ? Alignment.centerRight : Alignment.centerLeft,
      child: Container(
        margin: const EdgeInsets.symmetric(vertical: 4, horizontal: 8),
        padding: const EdgeInsets.all(12),
        decoration: BoxDecoration(
          color: isFromMe ? Colors.blue : Colors.grey[300],
          borderRadius: BorderRadius.circular(12),
        ),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          mainAxisSize: MainAxisSize.min,
          children: [
            Text(
              message.content,
              style: TextStyle(
                color: isFromMe ? Colors.white : Colors.black,
              ),
            ),
            const SizedBox(height: 4),
            Text(
              '${message.timestamp.hour}:${message.timestamp.minute.toString().padLeft(2, '0')}',
              style: TextStyle(
                fontSize: 12,
                color: isFromMe ? Colors.white70 : Colors.black54,
              ),
            ),
          ],
        ),
      ),
    );
  }

  /// Gets the color for the status bar based on connection state.
  Color _getStatusColor() {
    switch (_connectionState) {
      case OfflineConnectionState.connected:
        return Colors.green;
      case OfflineConnectionState.connecting:
      case OfflineConnectionState.scanning:
        return Colors.orange;
      case OfflineConnectionState.error:
        return Colors.red;
      case OfflineConnectionState.unavailable:
        return Colors.grey;
      default:
        return Colors.blue;
    }
  }

  /// Gets the icon for the status bar based on connection state.
  IconData _getStatusIcon() {
    switch (_connectionState) {
      case OfflineConnectionState.connected:
        return Icons.bluetooth_connected;
      case OfflineConnectionState.connecting:
        return Icons.bluetooth_searching;
      case OfflineConnectionState.scanning:
        return Icons.search;
      case OfflineConnectionState.error:
        return Icons.error;
      case OfflineConnectionState.unavailable:
        return Icons.bluetooth_disabled;
      default:
        return Icons.bluetooth;
    }
  }
} 