/// Represents the current state of a Bluetooth connection.
enum OfflineConnectionState {
  /// Bluetooth is not available or not enabled.
  unavailable,

  /// Bluetooth is available but not scanning.
  idle,

  /// Currently scanning for nearby devices.
  scanning,

  /// Connected to a device and ready for messaging.
  connected,

  /// Attempting to connect to a device.
  connecting,

  /// Disconnected from a device.
  disconnected,

  /// An error occurred during connection or messaging.
  error,
}

/// Extension to provide human-readable descriptions for connection states.
extension OfflineConnectionStateExtension on OfflineConnectionState {
  /// Returns a human-readable description of the connection state.
  String get description {
    switch (this) {
      case OfflineConnectionState.unavailable:
        return 'Bluetooth unavailable';
      case OfflineConnectionState.idle:
        return 'Ready to scan';
      case OfflineConnectionState.scanning:
        return 'Scanning for devices';
      case OfflineConnectionState.connected:
        return 'Connected';
      case OfflineConnectionState.connecting:
        return 'Connecting...';
      case OfflineConnectionState.disconnected:
        return 'Disconnected';
      case OfflineConnectionState.error:
        return 'Error occurred';
    }
  }

  /// Returns whether the state represents an active connection.
  bool get isConnected => this == OfflineConnectionState.connected;

  /// Returns whether the state represents an error condition.
  bool get isError => this == OfflineConnectionState.error;

  /// Returns whether the state represents an unavailable condition.
  bool get isUnavailable => this == OfflineConnectionState.unavailable;
} 