/// Represents a Bluetooth device discovered during scanning.
class OfflineDevice {
  /// The unique identifier for this device.
  final String id;

  /// The name of the device.
  final String name;

  /// The RSSI (signal strength) of the device.
  final int rssi;

  /// Whether this device is currently connected.
  final bool isConnected;

  /// Whether this device is connectable.
  final bool isConnectable;

  /// The timestamp when this device was discovered.
  final DateTime discoveredAt;

  /// Creates a new [OfflineDevice].
  const OfflineDevice({
    required this.id,
    required this.name,
    required this.rssi,
    required this.isConnected,
    required this.isConnectable,
    required this.discoveredAt,
  });

  /// Creates an [OfflineDevice] from a JSON map.
  factory OfflineDevice.fromJson(Map<String, dynamic> json) {
    return OfflineDevice(
      id: json['id'] as String,
      name: json['name'] as String,
      rssi: json['rssi'] as int,
      isConnected: json['isConnected'] as bool,
      isConnectable: json['isConnectable'] as bool,
      discoveredAt: DateTime.parse(json['discoveredAt'] as String),
    );
  }

  /// Converts this [OfflineDevice] to a JSON map.
  Map<String, dynamic> toJson() {
    return {
      'id': id,
      'name': name,
      'rssi': rssi,
      'isConnected': isConnected,
      'isConnectable': isConnectable,
      'discoveredAt': discoveredAt.toIso8601String(),
    };
  }

  /// Creates a copy of this [OfflineDevice] with the given fields replaced.
  OfflineDevice copyWith({
    String? id,
    String? name,
    int? rssi,
    bool? isConnected,
    bool? isConnectable,
    DateTime? discoveredAt,
  }) {
    return OfflineDevice(
      id: id ?? this.id,
      name: name ?? this.name,
      rssi: rssi ?? this.rssi,
      isConnected: isConnected ?? this.isConnected,
      isConnectable: isConnectable ?? this.isConnectable,
      discoveredAt: discoveredAt ?? this.discoveredAt,
    );
  }

  @override
  String toString() {
    return 'OfflineDevice(id: $id, name: $name, rssi: $rssi, isConnected: $isConnected, isConnectable: $isConnectable, discoveredAt: $discoveredAt)';
  }

  @override
  bool operator ==(Object other) {
    if (identical(this, other)) return true;
    return other is OfflineDevice &&
        other.id == id &&
        other.name == name &&
        other.rssi == rssi &&
        other.isConnected == isConnected &&
        other.isConnectable == isConnectable &&
        other.discoveredAt == discoveredAt;
  }

  @override
  int get hashCode {
    return id.hashCode ^
        name.hashCode ^
        rssi.hashCode ^
        isConnected.hashCode ^
        isConnectable.hashCode ^
        discoveredAt.hashCode;
  }
} 