/// Represents a message in the offline messaging system.
class OfflineMessage {
  /// The unique identifier for this message.
  final String id;

  /// The content of the message.
  final String content;

  /// The timestamp when the message was sent/received.
  final DateTime timestamp;

  /// Whether this message was sent by the current device.
  final bool isFromMe;

  /// The device ID of the sender.
  final String senderDeviceId;

  /// The device name of the sender.
  final String senderDeviceName;

  /// Creates a new [OfflineMessage].
  const OfflineMessage({
    required this.id,
    required this.content,
    required this.timestamp,
    required this.isFromMe,
    required this.senderDeviceId,
    required this.senderDeviceName,
  });

  /// Creates an [OfflineMessage] from a JSON map.
  factory OfflineMessage.fromJson(Map<String, dynamic> json) {
    return OfflineMessage(
      id: json['id'] as String,
      content: json['content'] as String,
      timestamp: DateTime.parse(json['timestamp'] as String),
      isFromMe: json['isFromMe'] as bool,
      senderDeviceId: json['senderDeviceId'] as String,
      senderDeviceName: json['senderDeviceName'] as String,
    );
  }

  /// Converts this [OfflineMessage] to a JSON map.
  Map<String, dynamic> toJson() {
    return {
      'id': id,
      'content': content,
      'timestamp': timestamp.toIso8601String(),
      'isFromMe': isFromMe,
      'senderDeviceId': senderDeviceId,
      'senderDeviceName': senderDeviceName,
    };
  }

  @override
  String toString() {
    return 'OfflineMessage(id: $id, content: $content, timestamp: $timestamp, isFromMe: $isFromMe, senderDeviceId: $senderDeviceId, senderDeviceName: $senderDeviceName)';
  }

  @override
  bool operator ==(Object other) {
    if (identical(this, other)) return true;
    return other is OfflineMessage &&
        other.id == id &&
        other.content == content &&
        other.timestamp == timestamp &&
        other.isFromMe == isFromMe &&
        other.senderDeviceId == senderDeviceId &&
        other.senderDeviceName == senderDeviceName;
  }

  @override
  int get hashCode {
    return id.hashCode ^
        content.hashCode ^
        timestamp.hashCode ^
        isFromMe.hashCode ^
        senderDeviceId.hashCode ^
        senderDeviceName.hashCode;
  }
} 