import 'dart:async';
import 'dart:convert';
import 'dart:math';

import 'package:flutter/foundation.dart';
import 'package:flutter_blue_plus/flutter_blue_plus.dart';
import 'package:permission_handler/permission_handler.dart';

import 'models/offline_connection_state.dart';
import 'models/offline_device.dart';
import 'models/offline_message.dart';

/// A Flutter package for peer-to-peer messaging via Bluetooth Low Energy (BLE).
/// 
/// This package enables devices to discover each other, establish connections,
/// and exchange text messages without requiring an internet connection.
/// 
/// Example usage:
/// ```dart
/// final offlineMessenger = OfflineMessenger();
/// 
/// // Listen to connection state changes
/// offlineMessenger.connectionStateStream.listen((state) {
///   print('Connection state: ${state.description}');
/// });
/// 
/// // Listen to discovered devices
/// offlineMessenger.discoveredDevicesStream.listen((devices) {
///   print('Found ${devices.length} devices');
/// });
/// 
/// // Listen to incoming messages
/// offlineMessenger.messagesStream.listen((message) {
///   print('Received: ${message.content}');
/// });
/// 
/// // Start scanning for devices
/// await offlineMessenger.startScanning();
/// 
/// // Connect to a device
/// await offlineMessenger.connectToDevice(device);
/// 
/// // Send a message
/// await offlineMessenger.sendMessage('Hello, world!');
/// ```
class OfflineMessenger {
  static const String _serviceUuid = '12345678-1234-1234-1234-123456789abc';
  static const String _characteristicUuid = '87654321-4321-4321-4321-cba987654321';

  final StreamController<OfflineConnectionState> _connectionStateController =
      StreamController<OfflineConnectionState>.broadcast();
  final StreamController<List<OfflineDevice>> _discoveredDevicesController =
      StreamController<List<OfflineDevice>>.broadcast();
  final StreamController<OfflineMessage> _messagesController =
      StreamController<OfflineMessage>.broadcast();

  BluetoothDevice? _connectedDevice;
  BluetoothCharacteristic? _messageCharacteristic;
  StreamSubscription<List<ScanResult>>? _scanSubscription;
  StreamSubscription<BluetoothConnectionState>? _connectionSubscription;
  StreamSubscription<List<int>>? _characteristicSubscription;
  bool _isScanning = false;
  bool _isAdvertising = false;
  final List<OfflineDevice> _discoveredDevices = [];
  final Map<String, BluetoothDevice> _bluetoothDevices = {};
  final Random _random = Random();

  /// Stream of connection state changes.
  Stream<OfflineConnectionState> get connectionStateStream =>
      _connectionStateController.stream;

  /// Stream of discovered devices.
  Stream<List<OfflineDevice>> get discoveredDevicesStream =>
      _discoveredDevicesController.stream;

  /// Stream of incoming and outgoing messages.
  Stream<OfflineMessage> get messagesStream => _messagesController.stream;

  /// Current connection state.
  OfflineConnectionState _connectionState = OfflineConnectionState.unavailable;

  /// Gets the current connection state.
  OfflineConnectionState get connectionState => _connectionState;

  /// Gets the currently connected device, if any.
  OfflineDevice? get connectedDevice {
    if (_connectedDevice == null) return null;
    return OfflineDevice(
      id: _connectedDevice!.remoteId.str,
      name: _connectedDevice!.platformName,
      rssi: 0,
      isConnected: true,
      isConnectable: true,
      discoveredAt: DateTime.now(),
    );
  }

  /// Gets the list of currently discovered devices.
  List<OfflineDevice> get discoveredDevices => List.unmodifiable(_discoveredDevices);

  /// Gets whether Bluetooth is currently scanning.
  bool get isScanning => _isScanning;

  /// Gets whether the device is currently advertising.
  bool get isAdvertising => _isAdvertising;

  /// Initializes the Bluetooth chat system.
  /// 
  /// This method should be called before using any other functionality.
  /// It checks for Bluetooth availability and requests necessary permissions.
  Future<bool> initialize() async {
    try {
      // Check if Bluetooth is available
      if (!await FlutterBluePlus.isSupported) {
        _updateConnectionState(OfflineConnectionState.unavailable);
        return false;
      }

      // Request permissions
      final bluetoothPermission = await Permission.bluetooth.request();
      final bluetoothScanPermission = await Permission.bluetoothScan.request();
      final bluetoothConnectPermission = await Permission.bluetoothConnect.request();

      if (bluetoothPermission.isDenied ||
          bluetoothScanPermission.isDenied ||
          bluetoothConnectPermission.isDenied) {
        _updateConnectionState(OfflineConnectionState.unavailable);
        return false;
      }

      // Check if Bluetooth is on
      if (await FlutterBluePlus.adapterState.first != BluetoothAdapterState.on) {
        _updateConnectionState(OfflineConnectionState.unavailable);
        return false;
      }

      _updateConnectionState(OfflineConnectionState.idle);
      return true;
    } catch (e) {
      if (kDebugMode) {
        print('OfflineMessenger: Error initializing: $e');
      }
      _updateConnectionState(OfflineConnectionState.error);
      return false;
    }
  }

  /// Starts scanning for nearby Bluetooth devices.
  /// 
  /// This method will discover devices that are advertising the OfflineMessenger service.
  /// The discovered devices will be emitted through the [discoveredDevicesStream].
  Future<void> startScanning() async {
    if (_isScanning) return;

    try {
      _updateConnectionState(OfflineConnectionState.scanning);
      _isScanning = true;
      _discoveredDevices.clear();
      _bluetoothDevices.clear();

      // Start scanning for devices with our service UUID
      await FlutterBluePlus.startScan(
        timeout: const Duration(seconds: 10),
        androidUsesFineLocation: true,
      );

      _scanSubscription = FlutterBluePlus.scanResults.listen((results) {
        for (final result in results) {
          final device = result.device;
          final rssi = result.rssi;

          // Check if device has our service
          if (result.advertisementData.serviceUuids
              .contains(Guid(_serviceUuid))) {
            final offlineDevice = OfflineDevice(
              id: device.remoteId.str,
              name: device.platformName.isNotEmpty
                  ? device.platformName
                  : 'Unknown Device',
              rssi: rssi,
              isConnected: false,
              isConnectable: result.advertisementData.connectable,
              discoveredAt: DateTime.now(),
            );

            // Store the BluetoothDevice for later connection
            _bluetoothDevices[device.remoteId.str] = device;

            // Update or add device to discovered list
            final existingIndex = _discoveredDevices
                .indexWhere((d) => d.id == offlineDevice.id);
            if (existingIndex >= 0) {
              _discoveredDevices[existingIndex] = offlineDevice;
            } else {
              _discoveredDevices.add(offlineDevice);
            }

            _discoveredDevicesController.add(List.unmodifiable(_discoveredDevices));
          }
        }
      });

      // Stop scanning after timeout
      Timer(const Duration(seconds: 10), () {
        stopScanning();
      });
    } catch (e) {
      if (kDebugMode) {
        print('OfflineMessenger: Error starting scan: $e');
      }
      _updateConnectionState(OfflineConnectionState.error);
      _isScanning = false;
    }
  }

  /// Stops scanning for devices.
  Future<void> stopScanning() async {
    if (!_isScanning) return;

    try {
      await FlutterBluePlus.stopScan();
      await _scanSubscription?.cancel();
      _scanSubscription = null;
      _isScanning = false;
      _updateConnectionState(OfflineConnectionState.idle);
    } catch (e) {
      if (kDebugMode) {
        print('OfflineMessenger: Error stopping scan: $e');
      }
    }
  }

  /// Connects to a specific device.
  /// 
  /// [device] should be a device discovered through scanning.
  /// Returns true if connection is successful.
  Future<bool> connectToDevice(OfflineDevice device) async {
    if (_connectedDevice != null) {
      await disconnect();
    }

    try {
      _updateConnectionState(OfflineConnectionState.connecting);

      // Get the BluetoothDevice from our stored map
      final bluetoothDevice = _bluetoothDevices[device.id];
      if (bluetoothDevice == null) {
        throw Exception('Device not found. Make sure to scan for devices first.');
      }

      // Connect to the device
      await bluetoothDevice.connect(timeout: const Duration(seconds: 10));

      // Discover services
      final services = await bluetoothDevice.discoverServices();
      final targetService = services.firstWhere(
        (service) => service.uuid == Guid(_serviceUuid),
        orElse: () => throw Exception('Service not found'),
      );

      // Get the characteristic for messaging
      _messageCharacteristic = targetService.characteristics.firstWhere(
        (char) => char.uuid == Guid(_characteristicUuid),
        orElse: () => throw Exception('Characteristic not found'),
      );

      // Subscribe to characteristic changes
      await _messageCharacteristic!.setNotifyValue(true);
      _characteristicSubscription = _messageCharacteristic!.lastValueStream.listen((value) {
        _handleIncomingMessage(value);
      });

      _connectedDevice = bluetoothDevice;
      _updateConnectionState(OfflineConnectionState.connected);

      // Listen for connection state changes
      _connectionSubscription = bluetoothDevice.connectionState.listen((state) {
        if (state == BluetoothConnectionState.disconnected) {
          _handleDisconnection();
        }
      });

      return true;
    } catch (e) {
      if (kDebugMode) {
        print('OfflineMessenger: Error connecting to device: $e');
      }
      _updateConnectionState(OfflineConnectionState.error);
      return false;
    }
  }

  /// Disconnects from the currently connected device.
  Future<void> disconnect() async {
    try {
      await _connectionSubscription?.cancel();
      await _characteristicSubscription?.cancel();
      await _connectedDevice?.disconnect();
      
      _connectedDevice = null;
      _messageCharacteristic = null;
      _connectionSubscription = null;
      _characteristicSubscription = null;
      
      _updateConnectionState(OfflineConnectionState.disconnected);
    } catch (e) {
      if (kDebugMode) {
        print('OfflineMessenger: Error disconnecting: $e');
      }
    }
  }

  /// Sends a message to the connected device.
  /// 
  /// [message] should be the text content to send.
  /// Returns true if the message was sent successfully.
  Future<bool> sendMessage(String message) async {
    if (_messageCharacteristic == null || _connectedDevice == null) {
      return false;
    }

    try {
      final offlineMessage = OfflineMessage(
        id: _generateMessageId(),
        content: message,
        timestamp: DateTime.now(),
        isFromMe: true,
        senderDeviceId: _connectedDevice!.remoteId.str,
        senderDeviceName: _connectedDevice!.platformName,
      );

      // Convert message to bytes
      final messageBytes = utf8.encode(jsonEncode(offlineMessage.toJson()));
      
      // Send the message
      await _messageCharacteristic!.write(messageBytes);

      // Emit the sent message
      _messagesController.add(offlineMessage);

      return true;
    } catch (e) {
      if (kDebugMode) {
        print('OfflineMessenger: Error sending message: $e');
      }
      return false;
    }
  }

  /// Starts advertising this device to be discoverable by others.
  /// 
  /// This allows other devices to find and connect to this device.
  Future<bool> startAdvertising() async {
    if (_isAdvertising) return true;

    try {
      // Note: This is a simplified implementation
      // In a real implementation, you would need to use platform-specific APIs
      // or a different package that supports BLE advertising
      _isAdvertising = true;
      return true;
    } catch (e) {
      if (kDebugMode) {
        print('OfflineMessenger: Error starting advertising: $e');
      }
      return false;
    }
  }

  /// Stops advertising this device.
  Future<void> stopAdvertising() async {
    _isAdvertising = false;
  }

  /// Disposes of all resources and cancels all subscriptions.
  Future<void> dispose() async {
    await stopScanning();
    await disconnect();
    await stopAdvertising();
    
    await _connectionStateController.close();
    await _discoveredDevicesController.close();
    await _messagesController.close();
  }

  /// Updates the connection state and notifies listeners.
  void _updateConnectionState(OfflineConnectionState newState) {
    _connectionState = newState;
    _connectionStateController.add(newState);
  }

  /// Handles incoming messages from the connected device.
  void _handleIncomingMessage(List<int> data) {
    try {
      final messageString = utf8.decode(data);
      final messageJson = jsonDecode(messageString) as Map<String, dynamic>;
      final message = OfflineMessage.fromJson(messageJson);
      
      // Mark as not from me since it's incoming
      final incomingMessage = OfflineMessage(
        id: message.id,
        content: message.content,
        timestamp: message.timestamp,
        isFromMe: false,
        senderDeviceId: message.senderDeviceId,
        senderDeviceName: message.senderDeviceName,
      );

      _messagesController.add(incomingMessage);
    } catch (e) {
      if (kDebugMode) {
        print('OfflineMessenger: Error handling incoming message: $e');
      }
    }
  }

  /// Handles disconnection events.
  void _handleDisconnection() {
    _connectedDevice = null;
    _messageCharacteristic = null;
    _updateConnectionState(OfflineConnectionState.disconnected);
  }

  /// Generates a unique message ID.
  String _generateMessageId() {
    return 'msg_${DateTime.now().millisecondsSinceEpoch}_${_random.nextInt(1000000)}';
  }
} 