import 'package:flutter_test/flutter_test.dart';
import 'package:offline_messenger/offline_messenger.dart';

void main() {
  group('OfflineMessenger Tests', () {
    late OfflineMessenger offlineMessenger;

    setUp(() {
      offlineMessenger = OfflineMessenger();
    });

    tearDown(() async {
      await offlineMessenger.dispose();
    });

    test('OfflineMessenger instance creation', () {
      expect(offlineMessenger, isNotNull);
      expect(offlineMessenger.connectionState, OfflineConnectionState.unavailable);
      expect(offlineMessenger.discoveredDevices, isEmpty);
      expect(offlineMessenger.isScanning, isFalse);
      expect(offlineMessenger.isAdvertising, isFalse);
    });

    test('OfflineMessage creation and serialization', () {
      final message = OfflineMessage(
        id: 'test_id',
        content: 'Hello, world!',
        timestamp: DateTime(2024, 1, 1, 12, 0, 0),
        isFromMe: true,
        senderDeviceId: 'device_1',
        senderDeviceName: 'Test Device',
      );

      expect(message.id, 'test_id');
      expect(message.content, 'Hello, world!');
      expect(message.isFromMe, isTrue);
      expect(message.senderDeviceId, 'device_1');
      expect(message.senderDeviceName, 'Test Device');

      // Test JSON serialization
      final json = message.toJson();
      expect(json['id'], 'test_id');
      expect(json['content'], 'Hello, world!');
      expect(json['isFromMe'], isTrue);

      // Test JSON deserialization
      final deserializedMessage = OfflineMessage.fromJson(json);
      expect(deserializedMessage.id, message.id);
      expect(deserializedMessage.content, message.content);
      expect(deserializedMessage.isFromMe, message.isFromMe);
    });

    test('OfflineDevice creation and serialization', () {
      final device = OfflineDevice(
        id: 'device_1',
        name: 'Test Device',
        rssi: -50,
        isConnected: false,
        isConnectable: true,
        discoveredAt: DateTime(2024, 1, 1, 12, 0, 0),
      );

      expect(device.id, 'device_1');
      expect(device.name, 'Test Device');
      expect(device.rssi, -50);
      expect(device.isConnected, isFalse);
      expect(device.isConnectable, isTrue);

      // Test JSON serialization
      final json = device.toJson();
      expect(json['id'], 'device_1');
      expect(json['name'], 'Test Device');
      expect(json['rssi'], -50);

      // Test JSON deserialization
      final deserializedDevice = OfflineDevice.fromJson(json);
      expect(deserializedDevice.id, device.id);
      expect(deserializedDevice.name, device.name);
      expect(deserializedDevice.rssi, device.rssi);
    });

    test('OfflineConnectionState enum values', () {
      expect(OfflineConnectionState.unavailable.description, 'Bluetooth unavailable');
      expect(OfflineConnectionState.idle.description, 'Ready to scan');
      expect(OfflineConnectionState.scanning.description, 'Scanning for devices');
      expect(OfflineConnectionState.connected.description, 'Connected');
      expect(OfflineConnectionState.connecting.description, 'Connecting...');
      expect(OfflineConnectionState.disconnected.description, 'Disconnected');
      expect(OfflineConnectionState.error.description, 'Error occurred');
    });

    test('OfflineConnectionState extension methods', () {
      expect(OfflineConnectionState.connected.isConnected, isTrue);
      expect(OfflineConnectionState.idle.isConnected, isFalse);
      expect(OfflineConnectionState.error.isError, isTrue);
      expect(OfflineConnectionState.unavailable.isUnavailable, isTrue);
    });

    test('OfflineDevice copyWith method', () {
      final originalDevice = OfflineDevice(
        id: 'device_1',
        name: 'Original Device',
        rssi: -50,
        isConnected: false,
        isConnectable: true,
        discoveredAt: DateTime(2024, 1, 1, 12, 0, 0),
      );

      final updatedDevice = originalDevice.copyWith(
        name: 'Updated Device',
        rssi: -60,
        isConnected: true,
      );

      expect(updatedDevice.id, originalDevice.id);
      expect(updatedDevice.name, 'Updated Device');
      expect(updatedDevice.rssi, -60);
      expect(updatedDevice.isConnected, isTrue);
      expect(updatedDevice.isConnectable, originalDevice.isConnectable);
      expect(updatedDevice.discoveredAt, originalDevice.discoveredAt);
    });

    test('OfflineMessage equality and hashCode', () {
      final message1 = OfflineMessage(
        id: 'test_id',
        content: 'Hello',
        timestamp: DateTime(2024, 1, 1, 12, 0, 0),
        isFromMe: true,
        senderDeviceId: 'device_1',
        senderDeviceName: 'Test Device',
      );

      final message2 = OfflineMessage(
        id: 'test_id',
        content: 'Hello',
        timestamp: DateTime(2024, 1, 1, 12, 0, 0),
        isFromMe: true,
        senderDeviceId: 'device_1',
        senderDeviceName: 'Test Device',
      );

      final message3 = OfflineMessage(
        id: 'different_id',
        content: 'Hello',
        timestamp: DateTime(2024, 1, 1, 12, 0, 0),
        isFromMe: true,
        senderDeviceId: 'device_1',
        senderDeviceName: 'Test Device',
      );

      expect(message1, equals(message2));
      expect(message1, isNot(equals(message3)));
      expect(message1.hashCode, equals(message2.hashCode));
      expect(message1.hashCode, isNot(equals(message3.hashCode)));
    });

    test('OfflineDevice equality and hashCode', () {
      final device1 = OfflineDevice(
        id: 'device_1',
        name: 'Test Device',
        rssi: -50,
        isConnected: false,
        isConnectable: true,
        discoveredAt: DateTime(2024, 1, 1, 12, 0, 0),
      );

      final device2 = OfflineDevice(
        id: 'device_1',
        name: 'Test Device',
        rssi: -50,
        isConnected: false,
        isConnectable: true,
        discoveredAt: DateTime(2024, 1, 1, 12, 0, 0),
      );

      final device3 = OfflineDevice(
        id: 'device_2',
        name: 'Test Device',
        rssi: -50,
        isConnected: false,
        isConnectable: true,
        discoveredAt: DateTime(2024, 1, 1, 12, 0, 0),
      );

      expect(device1, equals(device2));
      expect(device1, isNot(equals(device3)));
      expect(device1.hashCode, equals(device2.hashCode));
      expect(device1.hashCode, isNot(equals(device3.hashCode)));
    });

    test('OfflineMessage toString method', () {
      final message = OfflineMessage(
        id: 'test_id',
        content: 'Hello, world!',
        timestamp: DateTime(2024, 1, 1, 12, 0, 0),
        isFromMe: true,
        senderDeviceId: 'device_1',
        senderDeviceName: 'Test Device',
      );

      final string = message.toString();
      expect(string, contains('OfflineMessage'));
      expect(string, contains('test_id'));
      expect(string, contains('Hello, world!'));
      expect(string, contains('true'));
      expect(string, contains('device_1'));
      expect(string, contains('Test Device'));
    });

    test('OfflineDevice toString method', () {
      final device = OfflineDevice(
        id: 'device_1',
        name: 'Test Device',
        rssi: -50,
        isConnected: false,
        isConnectable: true,
        discoveredAt: DateTime(2024, 1, 1, 12, 0, 0),
      );

      final string = device.toString();
      expect(string, contains('OfflineDevice'));
      expect(string, contains('device_1'));
      expect(string, contains('Test Device'));
      expect(string, contains('-50'));
      expect(string, contains('false'));
      expect(string, contains('true'));
    });

    test('OfflineMessage fromJson with invalid data', () {
      expect(() => OfflineMessage.fromJson({}), throwsA(isA<TypeError>()));
      expect(() => OfflineMessage.fromJson({'id': 123}), throwsA(isA<TypeError>()));
    });

    test('OfflineDevice fromJson with invalid data', () {
      expect(() => OfflineDevice.fromJson({}), throwsA(isA<TypeError>()));
      expect(() => OfflineDevice.fromJson({'id': 123}), throwsA(isA<TypeError>()));
    });

    test('OfflineConnectionState all values', () {
      expect(OfflineConnectionState.values.length, 7);
      expect(OfflineConnectionState.values, contains(OfflineConnectionState.unavailable));
      expect(OfflineConnectionState.values, contains(OfflineConnectionState.idle));
      expect(OfflineConnectionState.values, contains(OfflineConnectionState.scanning));
      expect(OfflineConnectionState.values, contains(OfflineConnectionState.connected));
      expect(OfflineConnectionState.values, contains(OfflineConnectionState.connecting));
      expect(OfflineConnectionState.values, contains(OfflineConnectionState.disconnected));
      expect(OfflineConnectionState.values, contains(OfflineConnectionState.error));
    });
  });
} 