import 'package:flutter/material.dart';
import 'models/step_model.dart';
import 'widgets/spotlight.dart';
import 'widgets/tooltip.dart';

// Export public symbols
export 'models/step_model.dart';
export 'widgets/spotlight.dart';
export 'widgets/tooltip.dart';

class OnboardingTourController extends ChangeNotifier {
  _OnboardingTourState? _state;

  void _attach(_OnboardingTourState state) {
    _state = state;
  }

  void start() {
    _state?._startTour();
  }

  void stop() {
    _state?._stopTour();
  }
}

class OnboardingTour extends StatefulWidget {
  final List<TourStep> steps;
  final Widget child;
  final OnboardingTourController? controller;
  final bool darkTheme;
  final EdgeInsets spotlightPadding;
  final Duration animationDuration;
  final VoidCallback? onFinish;
  final VoidCallback? onSkip;

  const OnboardingTour({
    super.key,
    required this.steps,
    required this.child,
    this.controller,
    this.darkTheme = false,
    this.spotlightPadding = const EdgeInsets.all(8),
    this.animationDuration = const Duration(milliseconds: 300),
    this.onFinish,
    this.onSkip,
  });

  @override
  State<OnboardingTour> createState() => _OnboardingTourState();
}

class _OnboardingTourState extends State<OnboardingTour> {
  int? _currentStep;
  OverlayEntry? _overlayEntry;
  late OnboardingTourController _controller;

  @override
  void initState() {
    super.initState();
    _controller = widget.controller ?? OnboardingTourController();
    _controller._attach(this);
  }

  @override
  void dispose() {
    _removeOverlay();
    super.dispose();
  }

  void _startTour() {
    if (widget.steps.isEmpty) return;
    setState(() {
      _currentStep = 0;
    });
    _showOverlay();
  }

  void _stopTour() {
    setState(() {
      _currentStep = null;
    });
    _removeOverlay();
  }

  void _nextStep() {
    if (_currentStep == null) return;
    if (_currentStep! < widget.steps.length - 1) {
      setState(() {
        _currentStep = _currentStep! + 1;
      });
      _showOverlay();
    } else {
      widget.onFinish?.call();
      _stopTour();
    }
  }

  void _skipTour() {
    widget.onSkip?.call();
    _stopTour();
  }

  void _showOverlay() {
    WidgetsBinding.instance.addPostFrameCallback((_) {
      _removeOverlay();
      if (_currentStep == null) return;
      final step = widget.steps[_currentStep!];
      final context = step.key.currentContext;
      if (context == null) return;
      final box = context.findRenderObject() as RenderBox;
      final target = box.localToGlobal(Offset.zero);
      final rect = target & box.size;
      _overlayEntry = OverlayEntry(
        builder: (context) {
          return Stack(
            children: [
              Positioned.fill(
                child: Spotlight(
                  targetRect: rect.inflate(step.padding.horizontal / 2),
                  shape: step.shape,
                  borderRadius: 16,
                  color: widget.darkTheme
                      ? Colors.black.withValues(alpha: 0.7)
                      : Colors.white.withValues(alpha: 0.7),
                  animationDuration: widget.animationDuration,
                ),
              ),
              _buildTooltip(rect, step),
            ],
          );
        },
      );
      final overlay = Overlay.of(context, rootOverlay: true);
      overlay.insert(_overlayEntry!);
    });
  }

  Widget _buildTooltip(Rect rect, TourStep step) {
    Offset offset;
    switch (step.tooltipPosition) {
      case TooltipPosition.top:
        offset = Offset(rect.left, rect.top - 120);
        break;
      case TooltipPosition.bottom:
        offset = Offset(rect.left, rect.bottom + 12);
        break;
      case TooltipPosition.left:
        offset = Offset(rect.left - 220, rect.top);
        break;
      case TooltipPosition.right:
        offset = Offset(rect.right + 12, rect.top);
        break;
    }
    return Positioned(
      left: offset.dx,
      top: offset.dy,
      child: TourTooltip(
        title: step.title,
        description: step.description,
        position: step.tooltipPosition,
        isLast: _currentStep == widget.steps.length - 1,
        isDark: widget.darkTheme,
        onNext: _nextStep,
        onSkip: _skipTour,
        onFinish: widget.onFinish,
      ),
    );
  }

  void _removeOverlay() {
    _overlayEntry?.remove();
    _overlayEntry = null;
  }

  @override
  Widget build(BuildContext context) {
    return widget.child;
  }
}
