import 'package:flutter/material.dart';
import 'package:smart_device_info/smart_device_info.dart';

void main() {
  runApp(const SmartDeviceInfoApp());
}

/// Example app demonstrating the SmartDeviceInfo package capabilities.
class SmartDeviceInfoApp extends StatelessWidget {
  const SmartDeviceInfoApp({super.key});

  @override
  Widget build(BuildContext context) => MaterialApp(
        title: 'Smart Device Info Example',
        theme: ThemeData(
          primarySwatch: Colors.blue,
          useMaterial3: true,
        ),
        home: const DeviceInfoScreen(),
      );
}

/// Main screen displaying all device information.
class DeviceInfoScreen extends StatefulWidget {
  const DeviceInfoScreen({super.key});

  @override
  State<DeviceInfoScreen> createState() => _DeviceInfoScreenState();
}

class _DeviceInfoScreenState extends State<DeviceInfoScreen> {
  DeviceInfo? _deviceInfo;
  BatteryInfo? _batteryInfo;
  ScreenInfo? _screenInfo;
  CpuInfo? _cpuInfo;
  RamInfo? _ramInfo;
  bool _isLoading = true;

  @override
  void initState() {
    super.initState();
    _loadDeviceInfo();
  }

  Future<void> _loadDeviceInfo() async {
    setState(() {
      _isLoading = true;
    });

    try {
      final deviceInfo = await SmartDeviceInfo.instance.getDeviceInfo();
      final batteryInfo = await SmartDeviceInfo.instance.getBatteryInfo();
      final screenInfo = SmartDeviceInfo.instance.getScreenInfo();
      final cpuInfo = await SmartDeviceInfo.instance.getCpuInfo();
      final ramInfo = await SmartDeviceInfo.instance.getRamInfo();

      setState(() {
        _deviceInfo = deviceInfo;
        _batteryInfo = batteryInfo;
        _screenInfo = screenInfo;
        _cpuInfo = cpuInfo;
        _ramInfo = ramInfo;
        _isLoading = false;
      });
    } on Exception catch (e) {
      setState(() {
        _isLoading = false;
      });
      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text('Error loading device info: $e')),
        );
      }
    }
  }

  @override
  Widget build(BuildContext context) => Scaffold(
        appBar: AppBar(
          title: const Text('Smart Device Info'),
          backgroundColor: Theme.of(context).colorScheme.inversePrimary,
          actions: [
            TextButton(
              onPressed: () {
                SmartDeviceInfo.instance.clearCache();
                _loadDeviceInfo();
              },
              child: const Text('Refresh'),
            ),
          ],
        ),
        body: _isLoading
            ? const Center(child: CircularProgressIndicator())
            : SingleChildScrollView(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    _buildDeviceInfoCard(),
                    const SizedBox(height: 16),
                    _buildBatteryInfoCard(),
                    const SizedBox(height: 16),
                    _buildScreenInfoCard(),
                    const SizedBox(height: 16),
                    _buildCpuInfoCard(),
                    const SizedBox(height: 16),
                    _buildRamInfoCard(),
                  ],
                ),
              ),
      );

  Widget _buildDeviceInfoCard() {
    if (_deviceInfo == null) return const SizedBox.shrink();

    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(
              'Device Information',
              style: Theme.of(context).textTheme.titleLarge,
            ),
            const SizedBox(height: 16),
            _buildInfoRow('Manufacturer', _deviceInfo!.manufacturer ?? 'Unknown'),
            _buildInfoRow('Model', _deviceInfo!.model ?? 'Unknown'),
            _buildInfoRow('OS', 
                '${_deviceInfo!.osName ?? 'Unknown'} ${_deviceInfo!.osVersion ?? 'Unknown'}'),
            _buildInfoRow('Brand', _deviceInfo!.brand ?? 'Unknown'),
            _buildInfoRow('Product', _deviceInfo!.product ?? 'Unknown'),
            _buildInfoRow('Hardware', _deviceInfo!.hardware ?? 'Unknown'),
            _buildInfoRow('Device ID', _deviceInfo!.deviceId ?? 'Unknown'),
            _buildInfoRow('Build Number', _deviceInfo!.buildNumber ?? 'Unknown'),
            _buildInfoRow('SDK Version', _deviceInfo!.sdkVersion.toString()),
            _buildInfoRow('Emulator', _deviceInfo!.isEmulator ? 'Yes' : 'No'),
            if (_deviceInfo!.fingerprint != null)
              _buildInfoRow('Fingerprint', _deviceInfo!.fingerprint!),
            if (_deviceInfo!.serialNumber != null)
              _buildInfoRow('Serial Number', _deviceInfo!.serialNumber!),
          ],
        ),
      ),
    );
  }

  Widget _buildBatteryInfoCard() {
    if (_batteryInfo == null) return const SizedBox.shrink();

    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(
              'Battery Information',
              style: Theme.of(context).textTheme.titleLarge,
            ),
            const SizedBox(height: 16),
            _buildInfoRow('Level', _batteryInfo!.levelPercentage),
            _buildInfoRow('Charging', _batteryInfo!.isCharging ? 'Yes' : 'No'),
            _buildInfoRow('Available', _batteryInfo!.isAvailable ? 'Yes' : 'No'),
            if (_batteryInfo!.health != null)
              _buildInfoRow('Health', _batteryInfo!.health!),
            if (_batteryInfo!.temperature != null)
              _buildInfoRow('Temperature', 
                  '${_batteryInfo!.temperature!.toStringAsFixed(1)}°C'),
            if (_batteryInfo!.voltage != null)
              _buildInfoRow('Voltage', '${_batteryInfo!.voltage} mV'),
          ],
        ),
      ),
    );
  }

  Widget _buildScreenInfoCard() {
    if (_screenInfo == null) return const SizedBox.shrink();

    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(
              'Screen Information',
              style: Theme.of(context).textTheme.titleLarge,
            ),
            const SizedBox(height: 16),
            _buildInfoRow('Size', _screenInfo!.size),
            _buildInfoRow('Physical Size', _screenInfo!.physicalSize),
            _buildInfoRow('Pixel Ratio', _screenInfo!.pixelRatio.toStringAsFixed(2)),
            _buildInfoRow('Density', '${_screenInfo!.density.round()} DPI'),
            _buildInfoRow('Orientation', _screenInfo!.orientation),
            _buildInfoRow('Has Notch', _screenInfo!.hasNotch ? 'Yes' : 'No'),
            _buildInfoRow('Diagonal', 
                '${_screenInfo!.diagonalInches.toStringAsFixed(1)} inches'),
            if (_screenInfo!.refreshRate != null)
              _buildInfoRow('Refresh Rate', 
                  '${_screenInfo!.refreshRate!.toStringAsFixed(1)} Hz'),
            if (_screenInfo!.brightness != null)
              _buildInfoRow('Brightness', 
                  '${(_screenInfo!.brightness! * 100).round()}%'),
          ],
        ),
      ),
    );
  }

  Widget _buildCpuInfoCard() {
    if (_cpuInfo == null) return const SizedBox.shrink();

    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(
              'CPU Information',
              style: Theme.of(context).textTheme.titleLarge,
            ),
            const SizedBox(height: 16),
            _buildInfoRow('Architecture', _cpuInfo!.architecture),
            _buildInfoRow('Cores', _cpuInfo!.cores.toString()),
            _buildInfoRow('Available', _cpuInfo!.isAvailable ? 'Yes' : 'No'),
            if (_cpuInfo!.frequencyFormatted != null)
              _buildInfoRow('Frequency', _cpuInfo!.frequencyFormatted!),
            if (_cpuInfo!.model != null)
              _buildInfoRow('Model', _cpuInfo!.model!),
            if (_cpuInfo!.vendor != null)
              _buildInfoRow('Vendor', _cpuInfo!.vendor!),
            if (_cpuInfo!.usagePercentage != null)
              _buildInfoRow('Usage', _cpuInfo!.usagePercentage!),
            if (_cpuInfo!.temperature != null)
              _buildInfoRow('Temperature', 
                  '${_cpuInfo!.temperature!.toStringAsFixed(1)}°C'),
          ],
        ),
      ),
    );
  }

  Widget _buildRamInfoCard() {
    if (_ramInfo == null) return const SizedBox.shrink();

    return Card(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(
              'RAM Information',
              style: Theme.of(context).textTheme.titleLarge,
            ),
            const SizedBox(height: 16),
            _buildInfoRow('Total', _ramInfo!.totalFormatted),
            _buildInfoRow('Available', _ramInfo!.availableFormatted),
            _buildInfoRow('Used', _ramInfo!.usedFormatted),
            _buildInfoRow('Usage', _ramInfo!.usagePercentageFormatted),
            _buildInfoRow('Available', _ramInfo!.isAvailable ? 'Yes' : 'No'),
          ],
        ),
      ),
    );
  }

  Widget _buildInfoRow(String label, String value) => Padding(
        padding: const EdgeInsets.symmetric(vertical: 4),
        child: Row(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            SizedBox(
              width: 120,
              child: Text(
                label,
                style: const TextStyle(fontWeight: FontWeight.w500),
              ),
            ),
            Expanded(
              child: Text(
                value,
                style: const TextStyle(fontFamily: 'monospace'),
              ),
            ),
          ],
        ),
      );
} 