import 'package:flutter_test/flutter_test.dart';
import 'package:smart_device_info/smart_device_info.dart';

void main() {
  group('SmartDeviceInfo', () {
    test('should be a singleton', () {
      final instance1 = SmartDeviceInfo.instance;
      final instance2 = SmartDeviceInfo.instance;
      expect(identical(instance1, instance2), isTrue);
    });

    test('should clear cache', () {
      final instance = SmartDeviceInfo.instance;
      instance.clearCache();
      // No exception should be thrown
      expect(instance, isNotNull);
    });
  });

  group('DeviceInfo', () {
    test('should create from JSON', () {
      final json = {
        'manufacturer': 'Samsung',
        'model': 'Galaxy S21',
        'osName': 'Android',
        'osVersion': '13.0',
        'deviceId': 'test-device-id',
        'isEmulator': false,
        'brand': 'samsung',
        'product': 'SM-G991B',
        'hardware': 'qcom',
        'fingerprint': 'samsung/o1sxxx/o1s:13/TP1A.220624.014/S911BXXU3CWAF:user/release-keys',
        'serialNumber': 'ABC123DEF456',
        'buildNumber': 'TP1A.220624.014',
        'sdkVersion': 33,
      };

      final deviceInfo = DeviceInfo.fromJson(json);

      expect(deviceInfo.manufacturer, 'Samsung');
      expect(deviceInfo.model, 'Galaxy S21');
      expect(deviceInfo.osName, 'Android');
      expect(deviceInfo.osVersion, '13.0');
      expect(deviceInfo.deviceId, 'test-device-id');
      expect(deviceInfo.isEmulator, false);
      expect(deviceInfo.brand, 'samsung');
      expect(deviceInfo.product, 'SM-G991B');
      expect(deviceInfo.hardware, 'qcom');
      expect(deviceInfo.fingerprint, 'samsung/o1sxxx/o1s:13/TP1A.220624.014/S911BXXU3CWAF:user/release-keys');
      expect(deviceInfo.serialNumber, 'ABC123DEF456');
      expect(deviceInfo.buildNumber, 'TP1A.220624.014');
      expect(deviceInfo.sdkVersion, 33);
    });

    test('should convert to JSON', () {
      const deviceInfo = DeviceInfo(
        manufacturer: 'Apple',
        model: 'iPhone 14',
        osName: 'iOS',
        osVersion: '16.5',
        deviceId: 'test-ios-device',
        isEmulator: false,
        brand: 'apple',
        product: 'iPhone14,2',
        hardware: 'iPhone14,2',
        fingerprint: null,
        serialNumber: null,
        buildNumber: '20F75',
        sdkVersion: 16,
      );

      final json = deviceInfo.toJson();

      expect(json['manufacturer'], 'Apple');
      expect(json['model'], 'iPhone 14');
      expect(json['osName'], 'iOS');
      expect(json['osVersion'], '16.5');
      expect(json['deviceId'], 'test-ios-device');
      expect(json['isEmulator'], false);
      expect(json['brand'], 'apple');
      expect(json['product'], 'iPhone14,2');
      expect(json['hardware'], 'iPhone14,2');
      expect(json['fingerprint'], isNull);
      expect(json['serialNumber'], isNull);
      expect(json['buildNumber'], '20F75');
      expect(json['sdkVersion'], 16);
    });
  });

  group('BatteryInfo', () {
    test('should create from JSON', () {
      final json = {
        'level': 0.85,
        'isCharging': true,
        'health': 'Good',
        'temperature': 35.5,
        'voltage': 4200,
        'isAvailable': true,
      };

      final batteryInfo = BatteryInfo.fromJson(json);

      expect(batteryInfo.level, 0.85);
      expect(batteryInfo.isCharging, true);
      expect(batteryInfo.health, 'Good');
      expect(batteryInfo.temperature, 35.5);
      expect(batteryInfo.voltage, 4200);
      expect(batteryInfo.isAvailable, true);
      expect(batteryInfo.levelPercentage, '85%');
    });

    test('should convert to JSON', () {
      const batteryInfo = BatteryInfo(
        level: 0.75,
        isCharging: false,
        health: 'Good',
        temperature: 32,
        voltage: 3800,
        isAvailable: true,
      );

      final json = batteryInfo.toJson();

      expect(json['level'], 0.75);
      expect(json['isCharging'], false);
      expect(json['health'], 'Good');
      expect(json['temperature'], 32);
      expect(json['voltage'], 3800);
      expect(json['isAvailable'], true);
    });
  });

  group('ScreenInfo', () {
    test('should create from JSON', () {
      final json = {
        'width': 1080,
        'height': 1920,
        'physicalWidth': 1440,
        'physicalHeight': 2560,
        'pixelRatio': 3,
        'density': 480,
        'refreshRate': 60,
        'hasNotch': true,
        'orientation': 'portrait',
        'brightness': 0.8,
      };

      final screenInfo = ScreenInfo.fromJson(json);

      expect(screenInfo.width, 1080);
      expect(screenInfo.height, 1920);
      expect(screenInfo.physicalWidth, 1440);
      expect(screenInfo.physicalHeight, 2560);
      expect(screenInfo.pixelRatio, 3);
      expect(screenInfo.density, 480);
      expect(screenInfo.refreshRate, 60);
      expect(screenInfo.hasNotch, true);
      expect(screenInfo.orientation, 'portrait');
      expect(screenInfo.brightness, 0.8);
      expect(screenInfo.size, '1080x1920');
      expect(screenInfo.physicalSize, '1440x2560');
    });

    test('should convert to JSON', () {
      const screenInfo = ScreenInfo(
        width: 1080,
        height: 1920,
        physicalWidth: 1440,
        physicalHeight: 2560,
        pixelRatio: 3,
        density: 480,
        refreshRate: 60,
        hasNotch: true,
        orientation: 'portrait',
        brightness: 0.8,
      );

      final json = screenInfo.toJson();

      expect(json['width'], 1080);
      expect(json['height'], 1920);
      expect(json['physicalWidth'], 1440);
      expect(json['physicalHeight'], 2560);
      expect(json['pixelRatio'], 3);
      expect(json['density'], 480);
      expect(json['refreshRate'], 60);
      expect(json['hasNotch'], true);
      expect(json['orientation'], 'portrait');
      expect(json['brightness'], 0.8);
    });
  });

  group('CpuInfo', () {
    test('should create from JSON', () {
      final json = {
        'architecture': 'arm64',
        'cores': 8,
        'frequency': 2400000000,
        'model': 'Apple M1',
        'vendor': 'Apple',
        'usage': 0.45,
        'temperature': 45.5,
        'isAvailable': true,
      };

      final cpuInfo = CpuInfo.fromJson(json);

      expect(cpuInfo.architecture, 'arm64');
      expect(cpuInfo.cores, 8);
      expect(cpuInfo.frequency, 2400000000);
      expect(cpuInfo.model, 'Apple M1');
      expect(cpuInfo.vendor, 'Apple');
      expect(cpuInfo.usage, 0.45);
      expect(cpuInfo.temperature, 45.5);
      expect(cpuInfo.isAvailable, true);
      expect(cpuInfo.frequencyFormatted, '2.4 GHz');
      expect(cpuInfo.usagePercentage, '45%');
    });

    test('should convert to JSON', () {
      const cpuInfo = CpuInfo(
        architecture: 'arm64',
        cores: 8,
        frequency: 2400000000,
        model: 'Apple M1',
        vendor: 'Apple',
        usage: 0.45,
        temperature: 45.5,
        isAvailable: true,
      );

      final json = cpuInfo.toJson();

      expect(json['architecture'], 'arm64');
      expect(json['cores'], 8);
      expect(json['frequency'], 2400000000);
      expect(json['model'], 'Apple M1');
      expect(json['vendor'], 'Apple');
      expect(json['usage'], 0.45);
      expect(json['temperature'], 45.5);
      expect(json['isAvailable'], true);
    });
  });

  group('RamInfo', () {
    test('should create from JSON', () {
      final json = {
        'total': 8589934592, // 8 GB
        'available': 4294967296, // 4 GB
        'used': 4294967296, // 4 GB
        'usagePercentage': 0.5,
        'isAvailable': true,
      };

      final ramInfo = RamInfo.fromJson(json);

      expect(ramInfo.total, 8589934592);
      expect(ramInfo.available, 4294967296);
      expect(ramInfo.used, 4294967296);
      expect(ramInfo.usagePercentage, 0.5);
      expect(ramInfo.isAvailable, true);
      expect(ramInfo.totalFormatted, '8.0 GB');
      expect(ramInfo.availableFormatted, '4.0 GB');
      expect(ramInfo.usedFormatted, '4.0 GB');
      expect(ramInfo.usagePercentageFormatted, '50%');
    });

    test('should convert to JSON', () {
      const ramInfo = RamInfo(
        total: 8589934592, // 8 GB
        available: 4294967296, // 4 GB
        used: 4294967296, // 4 GB
        usagePercentage: 0.5,
        isAvailable: true,
      );

      final json = ramInfo.toJson();

      expect(json['total'], 8589934592);
      expect(json['available'], 4294967296);
      expect(json['used'], 4294967296);
      expect(json['usagePercentage'], 0.5);
      expect(json['isAvailable'], true);
    });
  });
} 