import 'package:flutter/material.dart';
import 'package:smart_form_validator_plus/smart_form_validator.dart';

void main() {
  // Example: override error messages for i18n
  ValidatorMessages.setCustomProvider((key, value, locale) {
    if (locale == 'vi') {
      switch (key) {
        case 'email':
          return 'Địa chỉ email không hợp lệ';
        case 'phone':
          return 'Số điện thoại không hợp lệ';
        case 'required':
          return 'Trường này là bắt buộc';
        case 'password_weak':
          return 'Mật khẩu quá yếu';
        case 'id':
          return 'Số căn cước/hộ chiếu không hợp lệ';
      }
    }
    // fallback to English
    return ValidatorMessages.get(key);
  });
  runApp(const MyApp());
}

class MyApp extends StatelessWidget {
  const MyApp({super.key});

  @override
  Widget build(BuildContext context) {
    return MaterialApp(
      title: 'Smart Form Validator Demo',
      theme: ThemeData(
        colorScheme: ColorScheme.fromSeed(seedColor: Colors.deepPurple),
      ),
      home: const DemoFormPage(),
    );
  }
}

class DemoFormPage extends StatefulWidget {
  const DemoFormPage({super.key});

  @override
  State<DemoFormPage> createState() => _DemoFormPageState();
}

class _DemoFormPageState extends State<DemoFormPage> {
  final _formKey = GlobalKey<FormState>();
  String _locale = 'en';

  final _emailController = TextEditingController();
  final _phoneController = TextEditingController();
  final _passwordController = TextEditingController();
  final _idController = TextEditingController();
  final _customController = TextEditingController();

  @override
  void dispose() {
    _emailController.dispose();
    _phoneController.dispose();
    _passwordController.dispose();
    _idController.dispose();
    _customController.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: const Text('Smart Form Validator Demo'),
        actions: [
          DropdownButton<String>(
            value: _locale,
            onChanged: (v) {
              if (v != null) setState(() => _locale = v);
            },
            items: const [
              DropdownMenuItem(
                value: 'en',
                child: Text('English'),
              ),
              DropdownMenuItem(
                value: 'vi',
                child: Text('Tiếng Việt'),
              ),
            ],
          ),
        ],
      ),
      body: Padding(
        padding: const EdgeInsets.all(16.0),
        child: Form(
          key: _formKey,
          child: ListView(
            children: [
              TextFormField(
                controller: _emailController,
                decoration: const InputDecoration(labelText: 'Email'),
                validator: (v) => Validator.email(v, locale: _locale),
              ),
              TextFormField(
                controller: _phoneController,
                decoration: const InputDecoration(labelText: 'Phone (E.164)'),
                validator: (v) => Validator.phone(v, locale: _locale),
              ),
              TextFormField(
                controller: _passwordController,
                decoration: const InputDecoration(labelText: 'Password'),
                obscureText: true,
                validator: (v) => Validator.password(v, locale: _locale, strict: true),
              ),
              TextFormField(
                controller: _idController,
                decoration: const InputDecoration(labelText: 'ID/Passport'),
                validator: (v) => Validator.id(v, locale: _locale, pattern: RegExp(r'^[A-Z0-9]{6,20} ')),
              ),
              TextFormField(
                controller: _customController,
                decoration: const InputDecoration(labelText: 'Custom: must be "flutter"'),
                validator: Validator.custom((value, {locale}) {
                  if (value != 'flutter') {
                    return locale == 'vi' ? 'Phải nhập flutter' : 'Must be flutter';
                  }
                  return null;
                }),
              ),
              const SizedBox(height: 24),
              ElevatedButton(
                onPressed: () {
                  if (_formKey.currentState?.validate() ?? false) {
                    ScaffoldMessenger.of(context).showSnackBar(
                      SnackBar(content: Text(_locale == 'vi' ? 'Hợp lệ!' : 'Valid!')),
                    );
                  }
                },
                child: Text(_locale == 'vi' ? 'Gửi' : 'Submit'),
              ),
            ],
          ),
        ),
      ),
    );
  }
}
