import 'rules.dart';
import 'localization.dart';

class Validator {
  static String? required(String? value, {String? locale}) {
    if (value == null || value.trim().isEmpty) {
      return ValidatorMessages.get('required', value: value, locale: locale);
    }
    return null;
  }

  static String? email(String? value, {String? locale}) {
    if (value == null || value.trim().isEmpty) return null;
    final emailRegex = RegExp(r'^[\w\.-]+@[\w\.-]+\.[a-zA-Z]{2,}');
    if (!emailRegex.hasMatch(value.trim())) {
      return ValidatorMessages.get('email', value: value, locale: locale);
    }
    return null;
  }

  static String? phone(String? value, {String? locale, String? countryCode}) {
    if (value == null || value.trim().isEmpty) return null;
    // Basic international phone regex (E.164)
    final phoneRegex = RegExp(r'^\+?[1-9]\d{7,14}');
    if (!phoneRegex.hasMatch(value.trim())) {
      return ValidatorMessages.get('phone', value: value, locale: locale);
    }
    return null;
  }

  static String? password(String? value, {String? locale, bool strict = false}) {
    if (value == null || value.isEmpty) return null;
    final weak = value.length < 6;
    final medium = value.length >= 6 && value.length < 10;
    final strong = value.length >= 10 &&
        RegExp(r'[A-Z]').hasMatch(value) &&
        RegExp(r'[a-z]').hasMatch(value) &&
        RegExp(r'[0-9]').hasMatch(value) &&
        RegExp(r'[!@#\$&*~]').hasMatch(value);
    if (weak) {
      return ValidatorMessages.get('password_weak', value: value, locale: locale);
    } else if (medium && strict) {
      return ValidatorMessages.get('password_medium', value: value, locale: locale);
    } else if (strong || (!strict && medium)) {
      return null;
    }
    return null;
  }

  static String? id(String? value, {String? locale, RegExp? pattern}) {
    if (value == null || value.trim().isEmpty) return null;
    final regex = pattern ?? RegExp(r'^[A-Za-z0-9]{6,20}');
    if (!regex.hasMatch(value.trim())) {
      return ValidatorMessages.get('id', value: value, locale: locale);
    }
    return null;
  }

  static ValidatorRule custom(ValidatorRule rule) => rule;
} 