import 'package:flutter/widgets.dart';

/// Default scaling rule based on width.
num _defaultScale(double width) {
  if (width < 600) return 1.0;
  if (width < 1200) return 1.2;
  return 1.5;
}

/// The scaling mode: by width or by height.
enum ScaleMode { width, height }

/// Extensions for responsive UI scaling on BuildContext.
extension UIScalerContext on BuildContext {
  /// Scale a value (e.g. size, spacing) according to the screen size.
  double scaled(num value, {ScaleMode mode = ScaleMode.width, num Function(double width)? customScale}) {
    final size = MediaQuery.of(this).size;
    final scale = customScale?.call(size.width) ?? _defaultScale(mode == ScaleMode.width ? size.width : size.height);
    return (value * scale).toDouble();
  }

  /// Scale a font size according to the screen size.
  double scaledFontSize(num value, {ScaleMode mode = ScaleMode.width, num Function(double width)? customScale}) {
    return scaled(value, mode: mode, customScale: customScale);
  }

  /// Scale EdgeInsets for padding according to the screen size.
  EdgeInsetsGeometry scaledPadding({num? all, num? horizontal, num? vertical, num? left, num? top, num? right, num? bottom, ScaleMode mode = ScaleMode.width, num Function(double width)? customScale}) {
    double s(num? v) => v == null ? 0 : scaled(v, mode: mode, customScale: customScale);
    if (all != null) return EdgeInsets.all(s(all));
    return EdgeInsets.only(
      left: s(left),
      top: s(top),
      right: s(right),
      bottom: s(bottom),
    ).add(EdgeInsets.symmetric(
      horizontal: s(horizontal),
      vertical: s(vertical),
    ));
  }

  /// Scale EdgeInsets for margin according to the screen size.
  EdgeInsetsGeometry scaledMargin({num? all, num? horizontal, num? vertical, num? left, num? top, num? right, num? bottom, ScaleMode mode = ScaleMode.width, num Function(double width)? customScale}) {
    return scaledPadding(
      all: all,
      horizontal: horizontal,
      vertical: vertical,
      left: left,
      top: top,
      right: right,
      bottom: bottom,
      mode: mode,
      customScale: customScale,
    );
  }
}

/// A widget like [SizedBox] but automatically scales width and height responsively.
class ScaledBox extends StatelessWidget {
  /// The width before scaling.
  final num? width;
  /// The height before scaling.
  final num? height;
  /// The child widget.
  final Widget? child;
  /// The scaling mode (by width or height).
  final ScaleMode mode;
  /// Optional custom scaling function.
  final num Function(double width)? customScale;

  /// Creates a [ScaledBox].
  /// [width] and [height] will be scaled automatically.
  /// [mode] controls scaling by width or height.
  /// [customScale] allows custom scaling logic.
  const ScaledBox({super.key, this.width, this.height, this.child, this.mode = ScaleMode.width, this.customScale});

  @override
  Widget build(BuildContext context) {
    return SizedBox(
      width: width != null ? context.scaled(width!, mode: mode, customScale: customScale) : null,
      height: height != null ? context.scaled(height!, mode: mode, customScale: customScale) : null,
      child: child,
    );
  }
}
